<?php /* =======================================================
|                                                              |
|   MiMiMi framework & CMS                                     |
|       Copyright 2022 MiMiMi Community                        |
|           > www.mimimi.software                              |
|       Licensed under CC BY 4                                 |
|           > www.creativecommons.org/licenses/by/4.0          |
|                                                              |
|   --------------------------------------------------------   |
|                                                              |
|   This file is a miniature library of helper routines        |
|   to bootstrap your application to the starting point        |
|   of the website.                                            |
|                                                              |
============================================================= */

    /* =========================================================
    |                                                          |
    |   If they are missing, define these main constants for   |
    |   the framework                                          |
    |                                                          |
    ========================================================= */

    define('MIMIMI_CLASS_PREFIX', 'Mimimi');

    $name = 'MIMIMI_CORE_VERSION';
    if (!defined($name)) {
        define($name, 'V.R - YYYY.MM.DD');
    }

    $name = 'MIMIMI_CORE_FOLDER';
    if (!defined($name)) {
        $folder = dirname(__FILE__);
        $folder = preg_replace('/^.*?([^\/\\]+)$/u', '$1', $folder);
        define($name, $folder . '/');
    }

    /* =========================================================
    |                                                          |
    |   If they are missing, define these constants for        |
    |   the modified version of the framework                  |
    |                                                          |
    ========================================================= */

    $name = 'MIMIMI_CMS_FOLDER';
    if (!defined($name)) {
        define($name, MIMIMI_CORE_FOLDER);
    }

    /* =========================================================
    |                                                          |
    |   Define a global routine to check for the existence     |
    |   of main constants                                      |
    |                                                          |
    |   ----------------------------------------------------   |
    |                                                          |
    |   Input parameters:                                      |
    |       NONE                                               |
    |                                                          |
    |   ----------------------------------------------------   |
    |                                                          |
    |   Output parameters:                                     |
    |       NONE                                               |
    |                                                          |
    ========================================================= */

    function mimimiCheck () {
        $plan = array(
            'MIMIMI_CLASS_PREFIX',
            'MIMIMI_CORE_FOLDER',
            'MIMIMI_CORE_VERSION'
        );
        foreach ($plan as $name) {
            if (!defined($name)) {
                mimimiStop('ERROR: Required constant ' . $name . ' not defined!');
            }
        }
    }

    /* =========================================================
    |                                                          |
    |   Define a global routine to stop execution and send     |
    |   a message to the user's browser with such HTTP status  |
    |   code                                                   |
    |                                                          |
    |   ----------------------------------------------------   |
    |                                                          |
    |   Input parameters:                                      |
    |       $message = STRING     = your message to the user   |
    |                  ARRAY OF URI (for status code 300)      |
    |                  URI          (for 301, 302, 307, 308)   |
    |                  RULES        (for 401, 407)             |
    |                  INTEGER      (for 429)                  |
    |       $code    = INTEGER = HTTP status code              |
    |                                                          |
    |   ----------------------------------------------------   |
    |                                                          |
    |   Output parameters:                                     |
    |       NONE                                               |
    |                                                          |
    ========================================================= */

    function mimimiStop ( $message, $code = 404 ) {
        $param = 'SERVER_PROTOCOL';
        $scheme = !empty($_SERVER[$param])
                  ? $_SERVER[$param]
                  : 'HTTP/1.1';
        switch ($code) {
            case 200:
                @ header($scheme . ' 200 OK', TRUE, $code);
                break;
            case 300:
                @ header($scheme . ' 300 Multiple Choices', TRUE, $code);
                if (is_array($message)) {
                    foreach ($message as $uri) {
                        @ header('Location: ' . $uri);
                    }
                } else {
                    @ header('Location: ' . $message);
                }
                exit;
            case 301:
                @ header($scheme . ' 301 Moved Permanently', TRUE, $code);
                @ header('Location: ' . $message);
                exit;
            case 302:
                @ header($scheme . ' 302 Found', TRUE, $code);
                @ header('Location: ' . $message);
                exit;
            case 307:
                @ header($scheme . ' 307 Temporary Redirect', TRUE, $code);
                @ header('Location: ' . $message);
                exit;
            case 308:
                @ header($scheme . ' 308 Permanent Redirect', TRUE, $code);
                @ header('Location: ' . $message);
                exit;
            case 400:
                @ header($scheme . ' 400 Bad Request', TRUE, $code);
                break;
            case 401:
                @ header($scheme . ' 401 Unauthorized', TRUE, $code);
                @ header('WWW-Authenticate: ' . $message);
                exit;
            case 403:
                @ header($scheme . ' 403 Forbidden', TRUE, $code);
                break;
            case 404:
                @ header($scheme . ' 404 Not Found', TRUE, $code);
                break;
            case 405:
                @ header($scheme . ' 405 Method Not Allowed', TRUE, $code);
                break;
            case 407:
                @ header($scheme . ' 407 Proxy Authentication Required', TRUE, $code);
                @ header('Proxy-Authenticate: ' . $message);
                exit;
            case 410:
                @ header($scheme . ' 410 Gone', TRUE, $code);
                break;
            case 429:
                @ header($scheme . ' 429 Too Many Requests', TRUE, $code);
                @ header('Retry-After: ' . $message);
                exit;
            case 451:
                @ header($scheme . ' 451 Unavailable For Legal Reasons', TRUE, $code);
                break;
            case 500:
                @ header($scheme . ' 500 Internal Server Error', TRUE, $code);
                break;
            case 501:
                @ header($scheme . ' 501 Not Implemented', TRUE, $code);
                break;
            case 503:
                @ header($scheme . ' 503 Service Unavailable', TRUE, $code);
                break;
        }
        echo $message;
        exit;
    }

    /* =========================================================
    |                                                          |
    |   Define a global routine to load a named module as      |
    |   the same property of the CMS object                    |
    |                                                          |
    |   ----------------------------------------------------   |
    |                                                          |
    |   Input parameters:                                      |
    |       $name    = STRING  = name of object property       |
    |       $cms     = OBJECT  = your application object       |
    |       $testing = BOOLEAN = TRUE if it is a test          |
    |                                                          |
    |   ----------------------------------------------------   |
    |                                                          |
    |   Output parameters:                                     |
    |       BOOLEAN = if it is a test                          |
    |       NONE    = if it isn't a test                       |
    |                                                          |
    ========================================================= */

    function mimimiCreate ( $name, $cms = NULL, $testing = FALSE ) {
        $path = dirname(__FILE__) . '/../';
        $extension = '.php';

        /* -------------------------------------
        |                                      |
        |   Convert a module name to CamelCase |
        |                                      |
        ------------------------------------- */

        $module = preg_replace('/[\W_]+/u', ' ', $name);
        $module = preg_replace('/(^\s+|\s+$)/u', '', $module);
        $module = mb_convert_case($module, MB_CASE_TITLE, 'UTF-8');
        $module = preg_replace('/\s/u', '', $module);

        /* -------------------------------------
        |                                      |
        |   Build a plan for loading           |
        |                                      |
        ------------------------------------- */

        $plan = array(
            MIMIMI_CMS_FOLDER  => 'My',
            MIMIMI_CORE_FOLDER => ''
        );

        /* -------------------------------------
        |                                      |
        |   Try to load the module by plan     |
        |                                      |
        ------------------------------------- */

        $isObject = !empty($cms);
        foreach ($plan as $folder => $class) {
            $folder .= $isObject
                       ? $module . '/'
                       : '';
            $file = $path . $folder . $module . $extension;
            if (file_exists($file)) {

                /* -----------------------------
                |                              |
                |   Try to test (if this mode) |
                |                              |
                ----------------------------- */

                $class .= MIMIMI_CLASS_PREFIX . $module;
                if ($testing) {
                    if (class_exists($class)) return TRUE;
                    $file = @ file_get_contents($file);
                    return is_string($file) &&
                           preg_match('/[\s;]class\s+' . $class . '[\s\{]/ui', ' ' . $file);
                }

                /* -----------------------------
                |                              |
                |   Try to create its object   |
                |                              |
                ----------------------------- */

                require_once($file);
                if (class_exists($class)) {
                    if ($isObject) {
                        $cms->$name = new $class($cms);
                        return $cms->$name;
                    }
                    return new $class();
                }

                /* -----------------------------
                |                              |
                |   Otherwise show error       |
                |                              |
                ----------------------------- */

                mimimiStop('ERROR: The ' . $module . $extension . ' file does not have the required class!');
            }
        }
        if ($testing) {
            return FALSE;
        }
        mimimiStop('ERROR: The ' . $module . $extension . ' file not found!');
    }

    /* =========================================================
    |                                                          |
    |   Block trailing injections                              |
    |                                                          |
    ========================================================= */

    return;