/* =============================================================
|                                                              |
|   LinkOfficer v.1.0                                          |
|       Author: MiMiMi Community                               |
|           > www.mimimi.software                              |
|       Licensed under CC BY 4                                 |
|           > www.creativecommons.org/licenses/by/4.0          |
|                                                              |
|   --------------------------------------------------------   |
|                                                              |
|   This file is used to serve private links located on the    |
|   client side of your website. Private links are specially   |
|   restricted links, the actual URI of which is returned      |
|   by the officer only at the moment of clicking on           |
|   the link.                                                  |
|                                                              |
|   --------------------------------------------------------   |
|                                                              |
|   For example, let's add a private link on the website page: |
|                                                              |
|       <a id="ID-FOR-OFFICER">Go Somewhere</a>                |
|                                                              |
|   --------------------------------------------------------   |
|                                                              |
|   For example, let's add this file after website footer:     |
|                                                              |
|       <script src="<?php theme() ?>js/link-officer.js">      |
|       </script>                                              |
|       <script>                                               |
|           cms.officer = new LinkOfficerClass();              |
|           cms.officer.init();                                |
|       </script>                                              |
|                                                              |
|   --------------------------------------------------------   |
|                                                              |
|   If you want to use the LinkOfficer component in            |
|   a specific website theme, for example YOUR-THEME-NAME,     |
|   then this file must be placed in the appropriate folder    |
|   of that theme.                                             |
|                                                              |
|   --------------------------------------------------------   |
|                                                              |
|   Note: This file is part of the LinkOfficer component       |
|         that consists these files:                           |
|             themes/YOUR-THEME-NAME/css/link-officer.css      |
|             themes/YOUR-THEME-NAME/js/link-officer.js        |
|             themes/YOUR-THEME-NAME/api/link-officer.php      |
|             YOUR-FRAMEWORK/LinkOfficer/LinkOfficer.php       |
|                                                              |
============================================================= */

    /* =========================================================
    |                                                          |
    |   We need to resolve the base URI later (see below       |
    |   how the private property SERVERBASE is formed).        |
    |   So we ask the CMS to store the URI of the current      |
    |   component. However, if this component is an inline     |
    |   javascript, that URI may be empty, and then the        |
    |   default URI from the CMS object will be taken as       |
    |   the base URI.                                          |
    |                                                          |
    ========================================================= */

    if (typeof cms == 'object') {
        cms.storeModuleUri(
            'LinkOfficer',
            document.currentScript.src
        );
    }

    /* =========================================================
    |                                                          |
    |   Now define the LinkOfficer pseudo class                |
    |                                                          |
    ========================================================= */

    function LinkOfficerClass () {
        var me = this;

        /* =====================================================
        |                                                      |
        |   Declare the public name of this component          |
        |                                                      |
        ===================================================== */

        me.name = 'LinkOfficer';

        /* =====================================================
        |                                                      |
        |   Declare the private properties                     |
        |                                                      |
        ===================================================== */

        var clickedNode    = null,
            flagBad        = 'data-ItWasCorrupted',
            flagDone       = 'data-ItWasHandled',
            errorPanel     = false,
            linkId         = '',
            serverBase     = hasCms()
                             ? cms.getModuleBase(me.name)
                             : '/',
            serverEndpoint = serverBase + 'api/link-officer';

        /* =====================================================
        |                                                      |
        |   Check the existence of the CMS object              |
        |                                                      |
        |   ------------------------------------------------   |
        |                                                      |
        |   Input parameters:                                  |
        |       NONE                                           |
        |                                                      |
        |   ------------------------------------------------   |
        |                                                      |
        |   Output parameters:                                 |
        |       BOOLEAN = TRUE  if that object exists          |
        |                 FALSE if it is absent                |
        |                                                      |
        ===================================================== */

        function hasCms () {
            return typeof cms == 'object';
        };

        /* =====================================================
        |                                                      |
        |   Stop an event propagation                          |
        |                                                      |
        |   ------------------------------------------------   |
        |                                                      |
        |   Input parameters:                                  |
        |       event = OBJECT = user's click info             |
        |                                                      |
        |   ------------------------------------------------   |
        |                                                      |
        |   Output parameters:                                 |
        |       NONE                                           |
        |                                                      |
        ===================================================== */

        function stopEvent ( event ) {
            event.stopImmediatePropagation();
            event.preventDefault();
        };

        /* =====================================================
        |                                                      |
        |   Build some parameters to query the server side     |
        |                                                      |
        |   ------------------------------------------------   |
        |                                                      |
        |   Input parameters:                                  |
        |       NONE                                           |
        |                                                      |
        |   ------------------------------------------------   |
        |                                                      |
        |   Output parameters:                                 |
        |       OBJECT = query parameters                      |
        |                                                      |
        ===================================================== */

        function makeQueryParams () {
            let data = new FormData();
            data.append('link', linkId);
            return {
                method: 'POST',
                body: data,
                mode: 'cors',
                cache: 'no-cache'
            };
        };

        /* =====================================================
        |                                                      |
        |   Get the active ErrorPanel component if it exists   |
        |   in the current website theme                       |
        |                                                      |
        |   ------------------------------------------------   |
        |                                                      |
        |   Input parameters:                                  |
        |       NONE                                           |
        |                                                      |
        |   ------------------------------------------------   |
        |                                                      |
        |   Output parameters:                                 |
        |       OBJECT = if that component exists              |
        |       NULL   = if component is absent                |
        |                                                      |
        ===================================================== */

        function getErrorPanel () {
            return hasCms()
                   ? cms.getModule('ErrorPanel')
                   : null;
        };

        /* =====================================================
        |                                                      |
        |   Display an error if it occurs on the server side   |
        |                                                      |
        |   ------------------------------------------------   |
        |                                                      |
        |   Input parameters:                                  |
        |       error = STRING = text to display               |
        |                                                      |
        |   ------------------------------------------------   |
        |                                                      |
        |   Output parameters:                                 |
        |       NONE                                           |
        |                                                      |
        ===================================================== */

        function onError ( error ) {
            if (clickedNode) {
                clickedNode.setAttribute(flagBad, true);
                clickedNode = null;
                if (errorPanel) {
                    errorPanel.display(error);
                }
            }
        };

        /* =====================================================
        |                                                      |
        |   Display a success state                            |
        |                                                      |
        |   ------------------------------------------------   |
        |                                                      |
        |   Input parameters:                                  |
        |       uri = STRING = page address                    |
        |                                                      |
        |   ------------------------------------------------   |
        |                                                      |
        |   Output parameters:                                 |
        |       NONE                                           |
        |                                                      |
        ===================================================== */

        function onSuccess ( uri ) {
            if (clickedNode) {
                if (errorPanel) {
                    errorPanel.display();
                }
                with (clickedNode) {
                    setAttribute('href', uri);
                    click();
                }
                clickedNode = null;
            }
        };

        /* =====================================================
        |                                                      |
        |   Handle an officer's response                       |
        |                                                      |
        |   ------------------------------------------------   |
        |                                                      |
        |   Input parameters:                                  |
        |       response = OBJECT = answer of the server side  |
        |                                                      |
        |   ------------------------------------------------   |
        |                                                      |
        |   Output parameters:                                 |
        |       NONE                                           |
        |                                                      |
        ===================================================== */

        function onResponse ( response ) {
            switch (response.status) {
                case 404:
                    onError('The serving component for this link is missing or disabled!');
                    break;
                case 200:
                    response.text().then(
                        function ( text ) {
                            let status = text.substr(0, 3);
                            switch (status) {
                                case 'OK ':
                                    text = text.substr(3);
                                    onSuccess(text);
                                    break;
                                case 'NO ':
                                    onError('Your current authorization rights do not allow you to access this document!');
                                    break;
                                default:
                                    onError('No URI for this link or it is invalid!');
                            }
                        }
                    );
                    break;
                default:
                    onError('An error "' + response.status + ' ' + response.statusText + '" has occured!');
            }
        };

        /* =====================================================
        |                                                      |
        |   Send the CLICK event to the server side            |
        |                                                      |
        |   ------------------------------------------------   |
        |                                                      |
        |   Input parameters:                                  |
        |       event = OBJECT = user's click info             |
        |                                                      |
        |   ------------------------------------------------   |
        |                                                      |
        |   Output parameters:                                 |
        |       NONE                                           |
        |                                                      |
        ===================================================== */

        function queryServer ( event ) {
            stopEvent(event);
            with (event) {
                clickedNode = target;
                target.removeEventListener('click', queryServer, true);
                linkId = target.getAttribute('id');
            }
            fetch(
                serverEndpoint,
                makeQueryParams()
            ).then(onResponse)
             .catch(onError);
        };

        /* =====================================================
        |                                                      |
        |   Charge the private links controlled by the officer |
        |                                                      |
        |   ------------------------------------------------   |
        |                                                      |
        |   Input parameters:                                  |
        |       selector = STRING = a DOM expression to        |
        |                           detect all private links   |
        |                           on the page                |
        |                                                      |
        |   ------------------------------------------------   |
        |                                                      |
        |   Output parameters:                                 |
        |       NONE                                           |
        |                                                      |
        ===================================================== */

        me.init = function ( selector = 'a[id]:not([href])' ) {
            let nodes = document.querySelectorAll(selector);
            if (nodes) {
                for (let i = 0; i < nodes.length; i++) {
                    with (nodes[i]) {
                        if (!hasAttribute(flagDone)) {
                            setAttribute(flagDone, true);
                            addEventListener('click', queryServer, true);
                        }
                    }
                }
                if (errorPanel === false) {
                    errorPanel = getErrorPanel();
                }
            }
        };
    };
