<?php /* =======================================================
|                                                              |
|   MiMiMi framework & CMS                                     |
|       Copyright 2022 MiMiMi Community                        |
|           > www.mimimi.software                              |
|       Licensed under CC BY 4                                 |
|           > www.creativecommons.org/licenses/by/4.0          |
|                                                              |
|   --------------------------------------------------------   |
|                                                              |
|   This file is a miniature library of helper routines        |
|   to bootstrap your application to the starting point        |
|   of the website.                                            |
|                                                              |
============================================================= */

    /* =========================================================
    |                                                          |
    |   Define some constants                                  |
    |                                                          |
    ========================================================= */

    define('MIMIMI_CLASS_PREFIX', 'Mimimi');

    /* =========================================================
    |                                                          |
    |   Define a global routine to check for the existence     |
    |   of main constants                                      |
    |                                                          |
    |   ----------------------------------------------------   |
    |                                                          |
    |   Input parameters:                                      |
    |       NONE                                               |
    |                                                          |
    |   ----------------------------------------------------   |
    |                                                          |
    |   Output parameters:                                     |
    |       NONE                                               |
    |                                                          |
    ========================================================= */

    function mimimiCheck () {
        $subject = 'Technical problem on [site]';

        /* -------------------------------------
        |                                      |
        |   Build a plan for testing           |
        |                                      |
        ------------------------------------- */

        $plan = array(
            'MIMIMI_CORE_VERSION'   => 'index.php',
            'MIMIMI_CORE_FOLDER'    => 'index.php',
            'MIMIMI_MODULES_FOLDER' => 'index.php',
            'MIMIMI_CMS_FOLDER'     => 'index.php',
            'MIMIMI_INSTALL_FOLDER' => 'index.php',

            'MIMIMI_DATABASE_DRIVER'       => 'Config.php',
            'MIMIMI_DATABASE_HOST'         => 'Config.php',
            'MIMIMI_DATABASE_PORT'         => 'Config.php',
            'MIMIMI_DATABASE_USER'         => 'Config.php',
            'MIMIMI_DATABASE_PASSWORD'     => 'Config.php',
            'MIMIMI_DATABASE_NAME'         => 'Config.php',
            'MIMIMI_DATABASE_TABLE_PREFIX' => 'Config.php',
            'MIMIMI_DATABASE_CHARSET'      => 'Config.php',
            'MIMIMI_DATABASE_COLLATION'    => 'Config.php',

            'MIMIMI_EMAILS_FOR_WEBMASTER'       => 'Config.php',
            'MIMIMI_EMAILS_FOR_CONTENT_MASTER'  => 'Config.php',
            'MIMIMI_EMAILS_FOR_TRAFFIC_MASTER'  => 'Config.php',
            'MIMIMI_EMAILS_FOR_SECURITY_MASTER' => 'Config.php',

            'MIMIMI_PHONES_FOR_WEBMASTER'       => 'Config.php',
            'MIMIMI_PHONES_FOR_CONTENT_MASTER'  => 'Config.php',
            'MIMIMI_PHONES_FOR_TRAFFIC_MASTER'  => 'Config.php',
            'MIMIMI_PHONES_FOR_SECURITY_MASTER' => 'Config.php',

            'MIMIMI_TELEGRAMS_FOR_WEBMASTER'       => 'Config.php',
            'MIMIMI_TELEGRAMS_FOR_CONTENT_MASTER'  => 'Config.php',
            'MIMIMI_TELEGRAMS_FOR_TRAFFIC_MASTER'  => 'Config.php',
            'MIMIMI_TELEGRAMS_FOR_SECURITY_MASTER' => 'Config.php',

            'MIMIMI_CLASS_PREFIX' => 'Routines.php'
        );

        /* -------------------------------------
        |                                      |
        |   Try to test the constant by plan   |
        |                                      |
        ------------------------------------- */

        foreach ($plan as $name => $file) {
            if (!defined($name)) {

                /* -----------------------------
                |                              |
                |   Inform your masters        |
                |                              |
                ----------------------------- */

                $msg = 'ERROR: Required constant ' . $name . ' at the file ' . $file . ' not defined!';
                $constant = 'MIMIMI_EMAILS_FOR_WEBMASTER';
                if (defined($constant)) {
                    mimimiInform(
                        $subject,
                        $msg,
                        constant($constant)
                    );
                }

                /* -----------------------------
                |                              |
                |   Inform the user            |
                |                              |
                ----------------------------- */

                mimimiStop($msg);
            }
        }
    }

    /* =========================================================
    |                                                          |
    |   Define a global routine to load a named module as      |
    |   the same property of the CMS object                    |
    |                                                          |
    |   ----------------------------------------------------   |
    |                                                          |
    |   Input parameters:                                      |
    |       $name    = STRING  = name of object property       |
    |       $cms     = OBJECT  = your application object       |
    |       $testing = BOOLEAN = TRUE if it is a test          |
    |                                                          |
    |   ----------------------------------------------------   |
    |                                                          |
    |   Output parameters:                                     |
    |       BOOLEAN = if it is a test                          |
    |       NONE    = if it isn't a test                       |
    |                                                          |
    ========================================================= */

    function mimimiCreate ( $name, $cms = NULL, $testing = FALSE ) {
        $constant = 'MIMIMI_EMAILS_FOR_WEBMASTER';
        $subject = 'Technical problem on [site]';

        $path = dirname(__FILE__) . '/../';
        $extension = '.php';

        /* -------------------------------------
        |                                      |
        |   Convert a module name to CamelCase |
        |                                      |
        ------------------------------------- */

        $module = preg_replace('/[\W_]+/u', ' ', $name);
        $module = preg_replace('/(^\s+|\s+$)/u', '', $module);
        $module = mb_convert_case($module, MB_CASE_TITLE, 'UTF-8');
        $module = preg_replace('/\s/u', '', $module);

        /* -------------------------------------
        |                                      |
        |   Build a plan for loading           |
        |                                      |
        ------------------------------------- */

        $plan = array(
            MIMIMI_INSTALL_FOLDER => 'My',
            MIMIMI_CMS_FOLDER     => 'My',
            MIMIMI_MODULES_FOLDER => '',
            MIMIMI_CORE_FOLDER    => ''
        );

        /* -------------------------------------
        |                                      |
        |   Try to load the module by plan     |
        |                                      |
        ------------------------------------- */

        $isObject = !empty($cms);
        foreach ($plan as $folder => $class) {
            $folder .= $isObject
                       ? $module . '/'
                       : '';
            $file = $path . $folder . $module . $extension;
            if (file_exists($file)) {

                /* -----------------------------
                |                              |
                |   Try to test (if this mode) |
                |                              |
                ----------------------------- */

                $class .= MIMIMI_CLASS_PREFIX . $module;
                if ($testing) {
                    if (class_exists($class)) return TRUE;
                    $file = @ file_get_contents($file);
                    return is_string($file) &&
                           preg_match('/[\s;]class\s+' . $class . '[\s\{]/ui', ' ' . $file);
                }

                /* -----------------------------
                |                              |
                |   Try to create its object   |
                |                              |
                ----------------------------- */

                try {
                    require_once($file);
                } catch (Exception $e) { }
                if (class_exists($class)) {
                    if ($isObject) {
                        $cms->$name = new $class($cms);
                        return $cms->$name;
                    }
                    return new $class();
                }

                /* -----------------------------
                |                              |
                |   Else infrom your masters   |
                |                              |
                ----------------------------- */

                $msg = 'ERROR: The ' . $module . $extension . ' file does not have the required class!';
                if (defined($constant)) {
                    mimimiInform(
                        $subject,
                        $msg,
                        constant($constant)
                    );
                }

                /* -----------------------------
                |                              |
                |   Show that error to user    |
                |                              |
                ----------------------------- */

                mimimiStop($msg);
            }
        }

        /* -------------------------------------
        |                                      |
        |   No more actions in test mode       |
        |                                      |
        ------------------------------------- */

        if ($testing) {
            return FALSE;
        }

        /* -------------------------------------
        |                                      |
        |   Otherwise inform your masters      |
        |                                      |
        ------------------------------------- */

        $msg = 'ERROR: The ' . $module . $extension . ' file not found!';
        if (defined($constant)) {
            mimimiInform(
                $subject,
                $msg,
                constant($constant)
            );
        }

        /* -------------------------------------
        |                                      |
        |   Show that error to user            |
        |                                      |
        ------------------------------------- */

        mimimiStop($msg);
    }

    /* =========================================================
    |                                                          |
    |   Define a global routine to load a required PHP file    |
    |                                                          |
    |   ----------------------------------------------------   |
    |                                                          |
    |   Input parameters:                                      |
    |       $filename = STRING  = relative path and name,      |
    |                             for example, Hello/Hello.php |
    |       $fromCore = BOOLEAN = TRUE  if look in CORE folder |
    |                           = FALSE if look in CMS folders |
    |                           = NULL  if look in both folders|
    |       $once     = BOOLEAN = TRUE  if only on first call  |
    |                           = FALSE if load on every call  |
    |                                                          |
    |   ----------------------------------------------------   |
    |                                                          |
    |   Output parameters:                                     |
    |       NONE                                               |
    |                                                          |
    ========================================================= */

    function mimimiInclude ( $filename, $fromCore = TRUE, $once = TRUE ) {
        $path = dirname(__FILE__) . '/../';

        /* -------------------------------------
        |                                      |
        |   Build a plan for loading           |
        |                                      |
        ------------------------------------- */

        $plan = array(
            MIMIMI_INSTALL_FOLDER => $fromCore !== TRUE,
            MIMIMI_CMS_FOLDER     => $fromCore !== TRUE,
            MIMIMI_MODULES_FOLDER => $fromCore !== FALSE,
            MIMIMI_CORE_FOLDER    => $fromCore !== FALSE
        );

        /* -------------------------------------
        |                                      |
        |   Loop through plan's items          |
        |                                      |
        ------------------------------------- */

        foreach ($plan as $folder => $enabled) {
            if ($enabled) {
                $file = $path . $folder . $filename;
                if (file_exists($file)) {

                    /* -------------------------
                    |                          |
                    |   Prepare input vars     |
                    |                          |
                    ------------------------- */

                    unset($path);
                    unset($enabled);
                    unset($fromCore);
                    unset($plan);
                    global $cms;

                    /* -------------------------
                    |                          |
                    |   Try to load the file   |
                    |                          |
                    ------------------------- */

                    if ($once) {
                        unset($once);
                        try {
                            include_once($file);
                        } catch (Exception $e) { }
                    } else {
                        unset($once);
                        try {
                            include($file);
                        } catch (Exception $e) { }
                    }
                    break;
                }
            }
        }
    }

    /* =========================================================
    |                                                          |
    |   Define a global routine to output a required THEME'S   |
    |   TEXTUAL file                                           |
    |                                                          |
    |   ----------------------------------------------------   |
    |                                                          |
    |   Input parameters:                                      |
    |       $filename = STRING  = relative path and name,      |
    |                             for example, css/style.css   |
    |       $fromCore = BOOLEAN = TRUE  if look in CORE folder |
    |                           = FALSE if look in CMS folder  |
    |                           = NULL  if look in both folders|
    |                                                          |
    |   ----------------------------------------------------   |
    |                                                          |
    |   Output parameters:                                     |
    |       NONE                                               |
    |                                                          |
    ========================================================= */

    function mimimiDump ( $filename, $fromCore = FALSE ) {
        $path = dirname(__FILE__) . '/../';
        $theme = 'Themes/' .
                 (
                     $fromCore
                     ? MIMIMI_CORE_THEME
                     : MIMIMI_CMS_THEME
                 ) .
                 '/';

        /* -------------------------------------
        |                                      |
        |   Build a plan for dumping           |
        |                                      |
        ------------------------------------- */

        $plan = array(
            MIMIMI_INSTALL_FOLDER => $fromCore !== TRUE,
            MIMIMI_CMS_FOLDER     => $fromCore !== TRUE,
            MIMIMI_MODULES_FOLDER => $fromCore !== FALSE,
            MIMIMI_CORE_FOLDER    => $fromCore !== FALSE
        );

        /* -------------------------------------
        |                                      |
        |   Loop through plan's items          |
        |                                      |
        ------------------------------------- */

        foreach ($plan as $folder => $enabled) {
            if ($enabled) {
                $file = $path . $folder . $theme . $filename;
                if (file_exists($file)) {

                    /* -------------------------
                    |                          |
                    |   Try to dump the file   |
                    |                          |
                    ------------------------- */

                    readfile($file);
                    break;
                }
            }
        }
    }

    /* =========================================================
    |                                                          |
    |   Define a global routine to load a required THEME'S     |
    |   TEMPLATE file                                          |
    |                                                          |
    |   ----------------------------------------------------   |
    |                                                          |
    |   Note: This routine should be considered as syntactic   |
    |         sugar to be used only in the theme files of      |
    |         the framework's modified version.                |
    |                                                          |
    |   ----------------------------------------------------   |
    |                                                          |
    |   Input parameters:                                      |
    |       $filename = STRING  = relative path and name, for  |
    |                             example, blocks/header.php   |
    |       $fromCore = BOOLEAN = TRUE  if look in CORE folder |
    |                           = FALSE if look in CMS folder  |
    |                                                          |
    |   ----------------------------------------------------   |
    |                                                          |
    |   Output parameters:                                     |
    |       NONE                                               |
    |                                                          |
    ========================================================= */

    function mimimiModule ( $filename, $fromCore = FALSE ) {
        $theme = 'Themes/' .
                 (
                     $fromCore
                     ? MIMIMI_CORE_THEME
                     : MIMIMI_CMS_THEME
                 ) .
                 '/';
        mimimiInclude(
            $theme . $filename,
            $fromCore == TRUE,
            FALSE
        );
    }

    /* =========================================================
    |                                                          |
    |   Define a global routine to inform your masters by      |
    |   email                                                  |
    |                                                          |
    |   ----------------------------------------------------   |
    |                                                          |
    |   Input parameters:                                      |
    |       $subject = STRING = title of the letter            |
    |       $message = STRING = your message to the master     |
    |       $to      = STRING = comma-separated list of emails |
    |       $details = ARRAY  = optional details               |
    |                                                          |
    |   ----------------------------------------------------   |
    |                                                          |
    |   Output parameters:                                     |
    |       NONE                                               |
    |                                                          |
    ========================================================= */

    function mimimiInform ( $subject, $message, $to, $details = array() ) {
    }

    /* =========================================================
    |                                                          |
    |   Define a global routine to retrieve a named COOKIE's   |
    |   parameter                                              |
    |                                                          |
    |   ----------------------------------------------------   |
    |                                                          |
    |   Input parameters:                                      |
    |       $param = STRING = name of the parameter            |
    |       $def   = MIXED  = a default value                  |
    |                                                          |
    |   ----------------------------------------------------   |
    |                                                          |
    |   Output parameters:                                     |
    |       MIXED = retrieved parameter                        |
    |                                                          |
    ========================================================= */

    function mimimiCookie ( $param, $def = NULL ) {
        return isset($_COOKIE[$param])
               ? $_COOKIE[$param]
               : $def;
    }

    /* =========================================================
    |                                                          |
    |   Define a global routine to retrieve a named ENV's      |
    |   parameter                                              |
    |                                                          |
    |   ----------------------------------------------------   |
    |                                                          |
    |   Input parameters:                                      |
    |       $param = STRING = name of the parameter            |
    |       $def   = MIXED  = a default value                  |
    |                                                          |
    |   ----------------------------------------------------   |
    |                                                          |
    |   Output parameters:                                     |
    |       MIXED = retrieved parameter                        |
    |                                                          |
    ========================================================= */

    function mimimiEnvironment ( $param, $def = '' ) {
        return isset($_ENV[$param])
               ? $_ENV[$param]
               : $def;
    }

    /* =========================================================
    |                                                          |
    |   Define a global routine to retrieve a named FILES's    |
    |   parameter                                              |
    |                                                          |
    |   ----------------------------------------------------   |
    |                                                          |
    |   Input parameters:                                      |
    |       $param = STRING = name of the parameter            |
    |       $def   = MIXED  = a default value                  |
    |                                                          |
    |   ----------------------------------------------------   |
    |                                                          |
    |   Output parameters:                                     |
    |       MIXED = retrieved parameter                        |
    |                                                          |
    ========================================================= */

    function mimimiFiles ( $param, $def = NULL ) {
        return isset($_FILES[$param])
               ? $_FILES[$param]
               : $def;
    }

    /* =========================================================
    |                                                          |
    |   Define a global routine to retrieve a named GET's      |
    |   parameter                                              |
    |                                                          |
    |   ----------------------------------------------------   |
    |                                                          |
    |   Input parameters:                                      |
    |       $param = STRING = name of the parameter            |
    |       $def   = MIXED  = a default value                  |
    |                                                          |
    |   ----------------------------------------------------   |
    |                                                          |
    |   Output parameters:                                     |
    |       MIXED = retrieved parameter                        |
    |                                                          |
    ========================================================= */

    function mimimiGet ( $param, $def = '' ) {
        return isset($_GET[$param])
               ? $_GET[$param]
               : $def;
    }

    /* =========================================================
    |                                                          |
    |   Define a global routine to retrieve a named POST's     |
    |   parameter                                              |
    |                                                          |
    |   ----------------------------------------------------   |
    |                                                          |
    |   Input parameters:                                      |
    |       $param = STRING = name of the parameter            |
    |       $def   = MIXED  = a default value                  |
    |                                                          |
    |   ----------------------------------------------------   |
    |                                                          |
    |   Output parameters:                                     |
    |       MIXED = retrieved parameter                        |
    |                                                          |
    ========================================================= */

    function mimimiPost ( $param, $def = '' ) {
        return isset($_POST[$param])
               ? $_POST[$param]
               : $def;
    }

    /* =========================================================
    |                                                          |
    |   Define a global routine to retrieve a named REQUEST's  |
    |   parameter                                              |
    |                                                          |
    |   ----------------------------------------------------   |
    |                                                          |
    |   Input parameters:                                      |
    |       $param = STRING = name of the parameter            |
    |       $def   = MIXED  = a default value                  |
    |                                                          |
    |   ----------------------------------------------------   |
    |                                                          |
    |   Output parameters:                                     |
    |       MIXED = retrieved parameter                        |
    |                                                          |
    ========================================================= */

    function mimimiRequest ( $param, $def = '' ) {
        return isset($_REQUEST[$param])
               ? $_REQUEST[$param]
               : $def;
    }

    /* =========================================================
    |                                                          |
    |   Define a global routine to retrieve a named SERVER's   |
    |   parameter                                              |
    |                                                          |
    |   ----------------------------------------------------   |
    |                                                          |
    |   Input parameters:                                      |
    |       $param = STRING = name of the parameter            |
    |       $def   = MIXED  = a default value                  |
    |                                                          |
    |   ----------------------------------------------------   |
    |                                                          |
    |   Output parameters:                                     |
    |       MIXED = retrieved parameter                        |
    |                                                          |
    ========================================================= */

    function mimimiServer ( $param, $def = '' ) {
        return isset($_SERVER[$param])
               ? $_SERVER[$param]
               : $def;
    }

    /* =========================================================
    |                                                          |
    |   Define a global routine to retrieve a named SESSION's  |
    |   parameter                                              |
    |                                                          |
    |   ----------------------------------------------------   |
    |                                                          |
    |   Input parameters:                                      |
    |       $param = STRING = name of the parameter            |
    |       $def   = MIXED  = a default value                  |
    |                                                          |
    |   ----------------------------------------------------   |
    |                                                          |
    |   Output parameters:                                     |
    |       MIXED = retrieved parameter                        |
    |                                                          |
    ========================================================= */

    function mimimiSession ( $param, $def = NULL ) {
        return isset($_SESSION[$param])
               ? $_SESSION[$param]
               : $def;
    }

    /* =========================================================
    |                                                          |
    |   Define a global routine to stop execution and send     |
    |   a message to the user's browser with such HTTP status  |
    |   code                                                   |
    |                                                          |
    |   ----------------------------------------------------   |
    |                                                          |
    |   Input parameters:                                      |
    |       $message = STRING     = your message to the user   |
    |                  ARRAY OF URI (for status code 300)      |
    |                  URI          (for 301, 302, 307, 308)   |
    |                  RULES        (for 401, 407)             |
    |                  INTEGER      (for 429)                  |
    |       $code    = INTEGER = HTTP status code              |
    |                                                          |
    |   ----------------------------------------------------   |
    |                                                          |
    |   Output parameters:                                     |
    |       NONE                                               |
    |                                                          |
    ========================================================= */

    function mimimiStop ( $message, $code = 404 ) {
        $scheme = mimimiServer(
            'SERVER_PROTOCOL',
            'HTTP/1.1'
        );
        switch ($code) {

            /* ---------------------------------
            |                                  |
            |   If a success code              |
            |                                  |
            --------------------------------- */

            case 200:
                @ header($scheme . ' 200 OK', TRUE, $code);
                break;

            /* ---------------------------------
            |                                  |
            |   If a redirect code             |
            |                                  |
            --------------------------------- */

            case 300:
                @ header($scheme . ' 300 Multiple Choices', TRUE, $code);
                if (is_array($message)) {
                    foreach ($message as $uri) {
                        @ header('Location: ' . $uri);
                    }
                } else {
                    @ header('Location: ' . $message);
                }
                exit;
            case 301:
                @ header($scheme . ' 301 Moved Permanently', TRUE, $code);
                @ header('Location: ' . $message);
                exit;
            case 302:
                @ header($scheme . ' 302 Found', TRUE, $code);
                @ header('Location: ' . $message);
                exit;
            case 307:
                @ header($scheme . ' 307 Temporary Redirect', TRUE, $code);
                @ header('Location: ' . $message);
                exit;
            case 308:
                @ header($scheme . ' 308 Permanent Redirect', TRUE, $code);
                @ header('Location: ' . $message);
                exit;

            /* ---------------------------------
            |                                  |
            |   If a content error code        |
            |                                  |
            --------------------------------- */

            case 400:
                @ header($scheme . ' 400 Bad Request', TRUE, $code);
                break;
            case 401:
                @ header($scheme . ' 401 Unauthorized', TRUE, $code);
                @ header('WWW-Authenticate: ' . $message);
                exit;
            case 403:
                @ header($scheme . ' 403 Forbidden', TRUE, $code);
                break;
            case 404:
                @ header($scheme . ' 404 Not Found', TRUE, $code);
                break;
            case 405:
                @ header($scheme . ' 405 Method Not Allowed', TRUE, $code);
                break;
            case 407:
                @ header($scheme . ' 407 Proxy Authentication Required', TRUE, $code);
                @ header('Proxy-Authenticate: ' . $message);
                exit;
            case 410:
                @ header($scheme . ' 410 Gone', TRUE, $code);
                break;
            case 429:
                @ header($scheme . ' 429 Too Many Requests', TRUE, $code);
                @ header('Retry-After: ' . $message);
                exit;
            case 451:
                @ header($scheme . ' 451 Unavailable For Legal Reasons', TRUE, $code);
                break;

            /* ---------------------------------
            |                                  |
            |   If a server error code         |
            |                                  |
            --------------------------------- */

            case 500:
                @ header($scheme . ' 500 Internal Server Error', TRUE, $code);
                break;
            case 501:
                @ header($scheme . ' 501 Not Implemented', TRUE, $code);
                break;
            case 503:
                @ header($scheme . ' 503 Service Unavailable', TRUE, $code);
                break;
        }

        /* -------------------------------------
        |                                      |
        |   And show message to user           |
        |                                      |
        ------------------------------------- */

        echo $message;
        exit;
    }

    /* =========================================================
    |                                                          |
    |   Define a global routine to display or retrieve the     |
    |   website domain's absolute URI                          |
    |                                                          |
    |   ----------------------------------------------------   |
    |                                                          |
    |   Input parameters:                                      |
    |       $toDisplay = BOOLEAN = TRUE if echo it to browser  |
    |                              FALSE if returns as result  |
    |                                                          |
    |   ----------------------------------------------------   |
    |                                                          |
    |   Output parameters:                                     |
    |       STRING = retrieved URI (with trailing slash)       |
    |                                                          |
    ========================================================= */

    function mimimiSite ( $toDisplay = TRUE ) {
        $domain = mimimiServer('HTTP_HOST');
        $scheme = mimimiServer('REQUEST_SCHEME');
        if ($scheme != '') {
            $scheme .= ':';
        }
        $value = $scheme . '//' . $domain . '/';
        if ($toDisplay) {
            echo $value;
        } else {
            return $value;
        }
    }

    /* =========================================================
    |                                                          |
    |   Define a global routine to display or retrieve the     |
    |   relative path's offset to the Core's folder            |
    |                                                          |
    |   ----------------------------------------------------   |
    |                                                          |
    |   Input parameters:                                      |
    |       $toDisplay = BOOLEAN = TRUE if echo it to browser  |
    |                              FALSE if returns as result  |
    |                                                          |
    |   ----------------------------------------------------   |
    |                                                          |
    |   Output parameters:                                     |
    |       STRING = retrieved path (without leading slash)    |
    |                                                          |
    ========================================================= */

    function mimimiRoot ( $toDisplay = TRUE ) {
        $value = mimimiServer('SCRIPT_NAME');
            $value = preg_replace('/[\s.]*[\/\\\\]+/u', '/', $value);
            $value = preg_replace('/\/+[^\/]+\.php$/ui', '/', $value);
            $value = preg_replace('/^\/+\s*/u', '', $value);
        if ($toDisplay) {
            echo $value;
        } else {
            return $value;
        }
    }

    /* =========================================================
    |                                                          |
    |   Define a global routine to display or retrieve the     |
    |   page's relative URI                                    |
    |                                                          |
    |   ----------------------------------------------------   |
    |                                                          |
    |   Input parameters:                                      |
    |       $toDisplay = BOOLEAN = TRUE if echo it to browser  |
    |                              FALSE if returns as result  |
    |                                                          |
    |   ----------------------------------------------------   |
    |                                                          |
    |   Output parameters:                                     |
    |       STRING = retrieved URI (without GET-parameters     |
    |                and leading or trailing slash)            |
    |                                                          |
    ========================================================= */

    function mimimiUri ( $toDisplay = TRUE ) {
        $encoding = 'UTF-8';
        $value = mimimiServer('REQUEST_URI');
            $value = preg_replace('/^([^\#?]*)[\#?].*$/u', '$1', $value);
            $value = preg_replace('/[\s.]*[\/\\\\]+/u', '/', $value);
            $value = preg_replace('/^\/+\s*/u', '', $value);
        $root = mimimiRoot(FALSE);
            $size = mb_strlen($root, $encoding);
            $test = mb_substr($value, 0, $size, $encoding);
            $test = mb_strtolower($test, $encoding) == mb_strtolower($root, $encoding);
            if ($test) {
                $value = mb_substr($value, $size, 1000000, $encoding);
            }
            $value = preg_replace('/\/+[\s.]*$/u', '', $value);
        if ($toDisplay) {
            echo $value;
        } else {
            return $value;
        }
    }

    /* =========================================================
    |                                                          |
    |   Block trailing injections                              |
    |                                                          |
    ========================================================= */

    return;