<?php /* =======================================================
|                                                              |
|   MiMiMi framework & CMS                                     |
|       Copyright 2022 MiMiMi Community                        |
|           > www.mimimi.software                              |
|       Licensed under CC BY 4                                 |
|           > www.creativecommons.org/licenses/by/4.0          |
|                                                              |
|   --------------------------------------------------------   |
|                                                              |
|   This file is a class for the DATABASE module. It is used   |
|   to read records from your database.                        |
|                                                              |
|   --------------------------------------------------------   |
|                                                              |
|   For example, let's get all records from EXAMPLE table:     |
|                                                              |
|       $cms = mimimiCreate('application');                    |
|       $result = $cms->db->query('SELECT * FROM __example');  |
|                                                              |
============================================================= */

    /* =========================================================
    |                                                          |
    |   Load the basis for system modules from the CORE folder |
    |                                                          |
    ========================================================= */

    mimimiInclude('Module.php');

    /* =========================================================
    |                                                          |
    |   Declare the child class                                |
    |                                                          |
    ========================================================= */

    class MimimiDb extends MimimiModule {
        protected $db = FALSE;

        /* =====================================================
        |                                                      |
        |   Perform some actions required to connect the       |
        |   database                                           |
        |                                                      |
        |   ------------------------------------------------   |
        |                                                      |
        |   Input parameters:                                  |
        |       $params = MIXED = some parameters if you need  |
        |                                                      |
        |   ------------------------------------------------   |
        |                                                      |
        |   Output parameters:                                 |
        |       MIXED = some result if you need                |
        |                                                      |
        ===================================================== */

        public function run ( $params = '' ) {
            if (empty($this->db)) {
                $subject = 'Problem of database configuration on [site]';
                $this->db = FALSE;

                /* -----------------------------
                |                              |
                |   Test the DB configuration  |
                |                              |
                ----------------------------- */

                $driver = MIMIMI_DATABASE_DRIVER;
                switch($driver) {
                    case '':
                        return;
                    case 'cubrid':
                    case 'dblib':
                    case 'firebird':
                    case 'ibm':
                    case     'odbc-ibm':
                    case 'informix':
                    case 'mssql':
                    case 'mysql':
                    case 'odbc':
                    case 'oci':
                    case 'oracle':
                    case 'pgsql':
                    case 'sqlsrv':
                    case 'sybase':
                        if (MIMIMI_DATABASE_HOST == '') {
                            $msg = 'ERROR: Database host is an empty string for the "' . $driver . '" driver!';
                            if ($this->cms->has->informer) {
                                $this->cms->informer->run(
                                    array(
                                        'subject'  => $subject,
                                        'message'  => $msg,
                                        'email'    => MIMIMI_EMAILS_FOR_WEBMASTER,
                                        'sms'      => MIMIMI_PHONES_FOR_WEBMASTER,
                                        'telegram' => MIMIMI_TELEGRAMS_FOR_WEBMASTER
                                    )
                                );
                            }
                            mimimiStop($msg);
                        }
                    case 'odbc':
                    case     'odbc-access':
                    case 'sqlite':
                        if (MIMIMI_DATABASE_NAME == '') {
                            $msg = 'ERROR: Database name is an empty string for the "' . $driver . '" driver!';
                            if ($this->cms->has->informer) {
                                $this->cms->informer->run(
                                    array(
                                        'subject'  => $subject,
                                        'message'  => $msg,
                                        'email'    => MIMIMI_EMAILS_FOR_WEBMASTER,
                                        'sms'      => MIMIMI_PHONES_FOR_WEBMASTER,
                                        'telegram' => MIMIMI_TELEGRAMS_FOR_WEBMASTER
                                    )
                                );
                            }
                            mimimiStop($msg);
                        }
                        break;
                    default:
                        $msg = 'ERROR: The "' . $driver . '" driver is no support!';
                        if ($this->cms->has->informer) {
                            $this->cms->informer->run(
                                array(
                                    'subject'  => $subject,
                                    'message'  => $msg,
                                    'email'    => MIMIMI_EMAILS_FOR_WEBMASTER,
                                    'sms'      => MIMIMI_PHONES_FOR_WEBMASTER,
                                    'telegram' => MIMIMI_TELEGRAMS_FOR_WEBMASTER
                                )
                            );
                        }
                        mimimiStop($msg);
                }

                /* -----------------------------
                |                              |
                |   Try to collect parameters  |
                |                              |
                ----------------------------- */

                switch($driver) {
                    case 'cubrid':
                    case 'dblib':
                    case 'mssql':
                    case 'mysql':
                    case 'pgsql':
                    case 'sybase':
                        $port = MIMIMI_DATABASE_PORT != ''
                                ? 'port=' . MIMIMI_DATABASE_PORT . ';'
                                : '';
                        $name = $driver . ':' .
                                'host=' . MIMIMI_DATABASE_HOST . ';' .
                                $port .
                                'dbname=' . MIMIMI_DATABASE_NAME . ';';
                        break;
                    case 'firebird':
                        $port = MIMIMI_DATABASE_PORT != ''
                                ? '/' . MIMIMI_DATABASE_PORT
                                : '';
                        $name = $driver . ':' .
                                'dbname=' . MIMIMI_DATABASE_HOST . $port . ':' .
                                            MIMIMI_DATABASE_NAME;
                        break;
                    case 'odbc-ibm':
                        $driver = 'odbc';
                    case 'ibm':
                        $port = MIMIMI_DATABASE_PORT != ''
                                ? 'PORT=' . MIMIMI_DATABASE_PORT . ';'
                                : '';
                        $name = $driver . ':' .
                                'DRIVER={IBM DB2 ODBC DRIVER};' .
                                'HOSTNAME=' . MIMIMI_DATABASE_HOST . ';' .
                                $port .
                                'DATABASE=' . MIMIMI_DATABASE_NAME . ';' .
                                'PROTOCOL=TCPIP;';
                        break;
                    case 'informix':
                        $port = MIMIMI_DATABASE_PORT != ''
                                ? ' service=' . MIMIMI_DATABASE_PORT . ';'
                                : '';
                        $name = $driver . ':' .
                                'host=' . MIMIMI_DATABASE_HOST . ';' .
                                $port .
                                ' database=' . MIMIMI_DATABASE_NAME . ';' .
                                ' server=ids_server;' .
                                ' protocol=onsoctcp;';
                        break;
                    case 'oracle':
                        $driver = 'oci';
                    case 'oci':
                        $port = MIMIMI_DATABASE_PORT != ''
                                ? ':' . MIMIMI_DATABASE_PORT
                                : '';
                        $name = $driver . ':' .
                                'dbname=//' . MIMIMI_DATABASE_HOST . $port . '/' .
                                              MIMIMI_DATABASE_NAME;
                        break;
                    case 'odbc':
                        $name = $driver . ':' .
                                MIMIMI_DATABASE_NAME;
                        break;
                    case 'odbc-access':
                        $name = 'odbc:' .
                                'Driver={Microsoft Access Driver (*.mdb)};' .
                                'Dbq=' . MIMIMI_DATABASE_NAME . ';';
                        break;
                    case 'sqlite':
                        $name = $driver . ':' .
                                MIMIMI_DATABASE_NAME;
                        break;
                    case 'sqlsrv':
                        $port = MIMIMI_DATABASE_PORT != ''
                                ? ',port=' . MIMIMI_DATABASE_PORT
                                : '';
                        $name = $driver . ':' .
                                'Server=' . MIMIMI_DATABASE_HOST . $port . ';' .
                                'Database=' . MIMIMI_DATABASE_NAME . ';';
                        break;
                }

                /* -----------------------------
                |                              |
                |   Try to connect             |
                |                              |
                ----------------------------- */

                try {
                    $this->db = new PDO(
                        $name,
                        MIMIMI_DATABASE_USER,
                        MIMIMI_DATABASE_PASSWORD
                    );
                    if (MIMIMI_DATABASE_CHARSET != '') {
                        $this->db->exec('SET CHARACTER SET "' . MIMIMI_DATABASE_CHARSET . '"');
                        if (MIMIMI_DATABASE_COLLATION != '') {
                            $this->db->exec('SET NAMES "' . MIMIMI_DATABASE_CHARSET . '" ' .
                                                'COLLATE "' . MIMIMI_DATABASE_COLLATION . '"');
                        }
                    }
                    $this->db->setAttribute(
                        PDO::ATTR_DEFAULT_FETCH_MODE,
                        PDO::FETCH_OBJ
                    );
                } catch (PDOException $error) {

                    /* -------------------------
                    |                          |
                    |   Else prepare email     |
                    |   details                |
                    |                          |
                    ------------------------- */

                    $msg = 'ERROR: No connection to the database.';
                    $details = array(
                        'Driver'   => MIMIMI_DATABASE_DRIVER,
                        'Host'     => MIMIMI_DATABASE_HOST,
                        'Port'     => MIMIMI_DATABASE_PORT,
                        'Database' => MIMIMI_DATABASE_NAME,
                        'User'     => MIMIMI_DATABASE_USER,
                        'Password' => '********** (see on the site)',
                        'Result'   => $error->getMessage()
                    );

                    /* -------------------------
                    |                          |
                    |   Inform your masters    |
                    |                          |
                    ------------------------- */

                    if ($this->cms->has->informer) {
                        $this->cms->informer->run(
                            array(
                                'subject'  => $subject,
                                'message'  => $msg,
                                'email'    => MIMIMI_EMAILS_FOR_WEBMASTER,
                                'sms'      => MIMIMI_PHONES_FOR_WEBMASTER,
                                'telegram' => MIMIMI_TELEGRAMS_FOR_WEBMASTER,
                                'details'  => $details
                            )
                        );
                    }

                    /* -------------------------
                    |                          |
                    |   Show the error to user |
                    |                          |
                    ------------------------- */

                    mimimiStop($msg);
                }
            }
        }

        /* =====================================================
        |                                                      |
        |   Execute an SQL statement and return the object     |
        |   associated with that statement                     |
        |                                                      |
        |   ------------------------------------------------   |
        |                                                      |
        |   Input parameters:                                  |
        |       $query  = STRING = the SQL statement           |
        |       $params = ARRAY  = values to insert instead    |
        |                          of QUESTION signs in the    |
        |                          SQL statement               |
        |                                                      |
        |   ------------------------------------------------   |
        |                                                      |
        |   Output parameters:                                 |
        |       OBJECT = if success                            |
        |       FALSE  = if failure                            |
        |                                                      |
        ===================================================== */

        public function query ( $query, $params = array() ) {
            $this->run();
            if (!empty($this->db)) {

                /* -----------------------------
                |                              |
                |   Set the table prefix       |
                |                              |
                ----------------------------- */

                $marker = '/([\s`,=\(\+-])__([a-z][a-z0-9_]*[\s`,.\);])/ui';
                $prefix = '$1' . MIMIMI_DATABASE_TABLE_PREFIX . '$2';
                $query = preg_replace($marker, $prefix, $query);

                /* -----------------------------
                |                              |
                |   Execute a query            |
                |                              |
                ----------------------------- */

                $object = $this->db->prepare($query);
                if ($object) {
                    if ($object->execute($params)) {

                        /* ---------------------
                        |                      |
                        |   Return the object  |
                        |                      |
                        --------------------- */

                        return $object;
                    }
                }
            }
            return FALSE;
        }
    };

    /* =========================================================
    |                                                          |
    |   Block trailing injections                              |
    |                                                          |
    ========================================================= */

    return;