<?php
/**
 * -------------------------------------------------------------------------
 * The HELPER module to provide generic routines for theme templates.
 *
 * -------------------------------------------------------------------------
 *
 * Implemented routines below are:
 *     getUrlScheme
 *     getThemeName
 *     sendStatus200
 *     sendStatus204
 *     sendStatus206
 *     sendStatus301
 *     sendStatus302
 *     sendStatus307
 *     sendStatus308
 *     sendStatus400
 *     sendStatus401
 *     sendStatus403
 *     sendStatus404
 *     sendStatus405
 *     sendStatus406
 *     sendStatus409
 *     sendStatus410
 *     sendStatus429
 *     sendStatus451
 *     sendStatus500
 *     sendStatus501
 *     sendStatus503
 *     sendHeaderHTML
 *     sendHeaderTEXT
 *     sendHeaderXML
 *     sendHeaderCSS
 *     sendHeaderJS
 *     sendHeaderJSON
 *     sendHeaderCSV
 *     sendHeaderEXCEL
 *     sendHeaderFILE
 *     sendHeaderExpires
 *     sendHeaderLastModified
 *     isEmpty
 *     isNonEmpty
 *     printValue
 *     printDomainUrl
 *     printSiteUrl
 *     printThemeUrl
 *     printPageUrl
 *
 * -------------------------------------------------------------------------
 *
 * @package    MimimiFramework
 * @subpackage Modules
 * @copyright  2022 MiMiMi Community
 *             https://mimimi.software/
 * @license    CC BY 4
 *             https://www.creativecommons.org/licenses/by/4.0
 * -------------------------------------------------------------------------
 */

mimimiInclude('Module.php');
class MimimiHelper extends MimimiModule {
};

/**
 * -------------------------------------------------------------------------
 * Retrieves the requested URL protocol.
 * -------------------------------------------------------------------------
 */

function getUrlScheme () {
    return mimimiServer('SERVER_PROTOCOL', 'HTTP/1.1');
}

/**
 * -------------------------------------------------------------------------
 * Retrieves the active theme name.
 *
 * @return  string
 * -------------------------------------------------------------------------
 */

function getThemeName () {
    return MIMIMI_CMS_THEME;
}

/**
 * -------------------------------------------------------------------------
 * Sends status code "200 OK" to the user's browser.
 * -------------------------------------------------------------------------
 */

function sendStatus200 () {
    $scheme = getUrlScheme();
    @ header($scheme . ' 200 OK', true, 200);
}

/**
 * -------------------------------------------------------------------------
 * Sends status code "204 No Content" to the user's browser.
 * -------------------------------------------------------------------------
 */

function sendStatus204 () {
    $scheme = getUrlScheme();
    @ header($scheme . ' 204 No Content', true, 204);
}

/**
 * -------------------------------------------------------------------------
 * Sends status code "206 Partial Content" to the user's browser.
 * -------------------------------------------------------------------------
 */

function sendStatus206 () {
    $scheme = getUrlScheme();
    @ header($scheme . ' 206 Partial Content', true, 206);
}

/**
 * -------------------------------------------------------------------------
 * Sends status code "301 Moved Permanently" to the user's browser.
 * -------------------------------------------------------------------------
 */

function sendStatus301 () {
    $scheme = getUrlScheme();
    @ header($scheme . ' 301 Moved Permanently', true, 301);
}

/**
 * -------------------------------------------------------------------------
 * Sends status code "302 Found" to the user's browser.
 * -------------------------------------------------------------------------
 */

function sendStatus302 () {
    $scheme = getUrlScheme();
    @ header($scheme . ' 302 Found', true, 302);
}

/**
 * -------------------------------------------------------------------------
 * Sends status code "307 Temporary Redirect" to the user's browser.
 * -------------------------------------------------------------------------
 */

function sendStatus307 () {
    $scheme = getUrlScheme();
    @ header($scheme . ' 307 Temporary Redirect', true, 307);
}

/**
 * -------------------------------------------------------------------------
 * Sends status code "308 Permanent Redirect" to the user's browser.
 * -------------------------------------------------------------------------
 */

function sendStatus308 () {
    $scheme = getUrlScheme();
    @ header($scheme . ' 308 Permanent Redirect', true, 308);
}

/**
 * -------------------------------------------------------------------------
 * Sends status code "400 Bad Request" to the user's browser.
 * -------------------------------------------------------------------------
 */

function sendStatus400 () {
    $scheme = getUrlScheme();
    @ header($scheme . ' 400 Bad Request', true, 400);
}

/**
 * -------------------------------------------------------------------------
 * Sends status code "401 Unauthorized" to the user's browser.
 * -------------------------------------------------------------------------
 */

function sendStatus401 () {
    $scheme = getUrlScheme();
    @ header($scheme . ' 401 Unauthorized', true, 401);
}

/**
 * -------------------------------------------------------------------------
 * Sends status code "403 Forbidden" to the user's browser.
 * -------------------------------------------------------------------------
 */

function sendStatus403 () {
    $scheme = getUrlScheme();
    @ header($scheme . ' 403 Forbidden', true, 403);
}

/**
 * -------------------------------------------------------------------------
 * Sends status code "404 Not Found" to the user's browser.
 * -------------------------------------------------------------------------
 */

function sendStatus404 () {
    $scheme = getUrlScheme();
    @ header($scheme . ' 404 Not Found', true, 404);
}

/**
 * -------------------------------------------------------------------------
 * Sends status code "405 Method Not Allowed" to the user's browser.
 * -------------------------------------------------------------------------
 */

function sendStatus405 () {
    $scheme = getUrlScheme();
    @ header($scheme . ' 405 Method Not Allowed', true, 405);
}

/**
 * -------------------------------------------------------------------------
 * Sends status code "406 Not Acceptable" to the user's browser.
 * -------------------------------------------------------------------------
 */

function sendStatus406 () {
    $scheme = getUrlScheme();
    @ header($scheme . ' 406 Not Acceptable', true, 406);
}

/**
 * -------------------------------------------------------------------------
 * Sends status code "409 Conflict" to the user's browser.
 * -------------------------------------------------------------------------
 */

function sendStatus409 () {
    $scheme = getUrlScheme();
    @ header($scheme . ' 409 Conflict', true, 409);
}

/**
 * -------------------------------------------------------------------------
 * Sends status code "410 Gone" to the user's browser.
 * -------------------------------------------------------------------------
 */

function sendStatus410 () {
    $scheme = getUrlScheme();
    @ header($scheme . ' 410 Gone', true, 410);
}

/**
 * -------------------------------------------------------------------------
 * Sends status code "429 Too Many Requests" to the user's browser.
 * -------------------------------------------------------------------------
 */

function sendStatus429 () {
    $scheme = getUrlScheme();
    @ header($scheme . ' 429 Too Many Requests', true, 429);
}

/**
 * -------------------------------------------------------------------------
 * Sends status code "451 Unavailable For Legal Reasons" to the user's browser.
 * -------------------------------------------------------------------------
 */

function sendStatus451 () {
    $scheme = getUrlScheme();
    @ header($scheme . ' 451 Unavailable For Legal Reasons', true, 451);
}

/**
 * -------------------------------------------------------------------------
 * Sends status code "500 Internal Server Error" to the user's browser.
 * -------------------------------------------------------------------------
 */

function sendStatus500 () {
    $scheme = getUrlScheme();
    @ header($scheme . ' 500 Internal Server Error', true, 500);
}

/**
 * -------------------------------------------------------------------------
 * Sends status code "501 Not Implemented" to the user's browser.
 * -------------------------------------------------------------------------
 */

function sendStatus501 () {
    $scheme = getUrlScheme();
    @ header($scheme . ' 501 Not Implemented', true, 501);
}

/**
 * -------------------------------------------------------------------------
 * Sends status code "503 Service Unavailable" to the user's browser.
 * -------------------------------------------------------------------------
 */

function sendStatus503 () {
    $scheme = getUrlScheme();
    @ header($scheme . ' 503 Service Unavailable', true, 503);
}

/**
 * -------------------------------------------------------------------------
 * Sends content type "Html" to the user's browser.
 *
 * @param  bool  $utf8  True if the HTML document is UTF-8 encoded.
 * -------------------------------------------------------------------------
 */

function sendHeaderHTML ( $utf8 = true ) {
    $charset = $utf8 ? '; charset=UTF-8' : '';
    @ header('Content-Type: text/html' . $charset, true);
}

/**
 * -------------------------------------------------------------------------
 * Sends content type "Text" to the user's browser.
 *
 * @param  bool  $utf8  True if the text document is UTF-8 encoded.
 * -------------------------------------------------------------------------
 */

function sendHeaderTEXT ( $utf8 = true ) {
    $charset = $utf8 ? '; charset=UTF-8' : '';
    @ header('Content-Type: text/plain' . $charset, true);
}

/**
 * -------------------------------------------------------------------------
 * Sends content type "Xml" to the user's browser.
 *
 * @param  bool  $utf8  True if the XML document is UTF-8 encoded.
 * -------------------------------------------------------------------------
 */

function sendHeaderXML ( $utf8 = true ) {
    $charset = $utf8 ? '; charset=UTF-8' : '';
    @ header('Content-Type: text/xml' . $charset, true);
}

/**
 * -------------------------------------------------------------------------
 * Sends content type "Style" to the user's browser.
 *
 * @param  bool  $utf8  True if the style file is UTF-8 encoded.
 * -------------------------------------------------------------------------
 */

function sendHeaderCSS ( $utf8 = true ) {
    $charset = $utf8 ? '; charset=UTF-8' : '';
    @ header('Content-Type: text/css' . $charset, true);
}

/**
 * -------------------------------------------------------------------------
 * Sends content type "Javascript" to the user's browser.
 *
 * @param  bool  $utf8  True if javascript file is UTF-8 encoded.
 * -------------------------------------------------------------------------
 */

function sendHeaderJS ( $utf8 = true ) {
    $charset = $utf8 ? '; charset=UTF-8' : '';
    @ header('Content-Type: application/javascript' . $charset, true);
}

/**
 * -------------------------------------------------------------------------
 * Sends content type "Json" to the user's browser.
 *
 * @param  bool  $utf8  True if the JSON file is UTF-8 encoded.
 * -------------------------------------------------------------------------
 */

function sendHeaderJSON ( $utf8 = true ) {
    $charset = $utf8 ? '; charset=UTF-8' : '';
    @ header('Content-Type: application/json' . $charset, true);
}

/**
 * -------------------------------------------------------------------------
 * Sends content type "Csv" to the user's browser.
 *
 * @param  bool  $utf8  True if the CSV file is UTF-8 encoded.
 * -------------------------------------------------------------------------
 */

function sendHeaderCSV ( $utf8 = true ) {
    $charset = $utf8 ? '; charset=UTF-8' : '';
    @ header('Content-Type: text/csv' . $charset, true);
}

/**
 * -------------------------------------------------------------------------
 * Sends content type "Excel" to the user's browser.
 *
 * @param  bool  $utf8  True if the XLS file is UTF-8 encoded.
 * -------------------------------------------------------------------------
 */

function sendHeaderEXCEL ( $utf8 = true ) {
    $charset = $utf8 ? '; charset=UTF-8' : '';
    @ header('Content-Type: application/vnd.ms-excel' . $charset, true);
}

/**
 * -------------------------------------------------------------------------
 * Sends content type "File" to the user's browser.
 *
 * @param  string  $filename  The name that the file will have after downloading.
 * @param  bool    $inlined   True if the file is an attachment (downloadable file),
 *                            False if it is an inline document.
 * -------------------------------------------------------------------------
 */

function sendHeaderFILE ( $filename, $inlined = false ) {
    $type = $inlined ? 'inline' : 'attachment';
    @ header('Content-Description: File Transfer', true);
    @ header('Content-Disposition: ' . $type . '; filename="' . $filename . '"', true);
    @ header('Content-Transfer-Encoding: binary', true);
}

/**
 * -------------------------------------------------------------------------
 * Sends header "Expires" to the user's browser.
 *
 * @param  int  $plus  Time offset (in hours) relative to server time.
 * -------------------------------------------------------------------------
 */

function sendHeaderExpires ( $plus = 1 ) {
    $time = time() + $plus * 3600;
    @ header('Expires: ' . date('r', $time), true);
    @ header('Cache-Control: public', true);
    @ header('Pragma: public', true);
}

/**
 * -------------------------------------------------------------------------
 * Sends header "Last-Modified" to the user's browser.
 *
 * @param  int  $plus  Time offset (in hours) relative to server time.
 * -------------------------------------------------------------------------
 */

function sendHeaderLastModified ( $plus = 0 ) {
    $time = time() + $plus * 3600;
    @ header('Last-Modified: ' . date('r', $time), true);
}

/**
 * -------------------------------------------------------------------------
 * Checks if the value is empty.
 *
 * @param   string  $value  Value to be checked.
 * @return  bool
 * -------------------------------------------------------------------------
 */

function isEmpty ( $value ) {
    return empty($value) ||
           preg_replace('~(^\s+|\s+$)~u', '', $value) == '';
}

/**
 * -------------------------------------------------------------------------
 * Checks if the value is non-empty.
 *
 * @param   string  $value  Value to be checked.
 * @return  bool
 * -------------------------------------------------------------------------
 */

function isNonEmpty ( $value ) {
    return ! isEmpty($value);
}

/**
 * -------------------------------------------------------------------------
 * Safely prints value.
 *
 * @param   string  $value  Value to be printed.
 * @param   bool    $echo   True if sent to the browser,
 *                          False if returned as a result.
 * @return  string
 * -------------------------------------------------------------------------
 */

function printValue ( $value, $echo = true ) {
    $value = htmlspecialchars($value, ENT_QUOTES, 'UTF-8');
    if ($echo) {
        echo $value;
        return '';
    }
    return $value;
}

/**
 * -------------------------------------------------------------------------
 * Prints the URL of the domain (physical root of the website).
 *
 * @param   bool    $echo      True if sent to the browser,
 *                             False if returned as a result.
 * @param   bool    $relative  True if the relative path is printed (it's an empty string),
 *                             False if the absolute path is printed.
 * @return  string
 * -------------------------------------------------------------------------
 */

function printDomainUrl ( $echo = true, $relative = false ) {
    if ($relative) return '';
    if ($echo) {
        mimimiSite();
        return '';
    }
    return mimimiSite(false);
}

/**
 * -------------------------------------------------------------------------
 * Prints the URL of the website root.
 *
 * @param   bool    $echo      True if sent to the browser,
 *                             False if returned as a result.
 * @param   bool    $relative  True if the relative path is printed (it's an empty string),
 *                             False if the absolute path is printed.
 * @return  string
 * -------------------------------------------------------------------------
 */

function printSiteUrl ( $echo = true, $relative = false ) {
    if ($relative) return '';
    if ($echo) {
        printDomainUrl();
        mimimiRoot();
        return '';
    }
    return printDomainUrl(false) .
           mimimiRoot(false);
}

/**
 * -------------------------------------------------------------------------
 * Prints the root URL of the website's active theme.
 *
 * @param   bool    $echo      True if sent to the browser,
 *                             False if returned as a result.
 * @param   bool    $relative  True if the relative path is printed,
 *                             False if the absolute path is printed.
 * @return  string
 * -------------------------------------------------------------------------
 */

function printThemeUrl ( $echo = true, $relative = false ) {
    $folder = MIMIMI_CMS_FOLDER . 'Themes/' . getThemeName() . '/';
    return printSiteUrl($echo, $relative) .
           printValue($folder, $echo);
}

/**
 * -------------------------------------------------------------------------
 * Prints the URL of the current web page.
 *
 * @param   bool    $echo      True if sent to the browser,
 *                             False if returned as a result.
 * @param   bool    $relative  True if the relative path is printed,
 *                             False if the absolute path is printed.
 * @return  string
 * -------------------------------------------------------------------------
 */

function printPageUrl ( $echo = true, $relative = false ) {
    if ($echo) {
        printSiteUrl(true, $relative);
        mimimiUri();
        return '';
    }
    return printSiteUrl(false, $relative) .
           mimimiUri(false);
}
