<?php /* =======================================================
|                                                              |
|   MiMiMi framework & CMS                                     |
|       Copyright 2022 MiMiMi Community                        |
|           > www.mimimi.software                              |
|       Licensed under CC BY 4                                 |
|           > www.creativecommons.org/licenses/by/4.0          |
|                                                              |
|   --------------------------------------------------------   |
|                                                              |
|   This file is a class for the VISITOR module. It retrieves  |
|   the data of the current visitor.                           |
|                                                              |
|   --------------------------------------------------------   |
|                                                              |
|   For example, let's fetch the current visitor's data:       |
|                                                              |
|       $cms = mimimiCreate('application');                    |
|       $data = $cms->visitor->run();                          |
|                                                              |
============================================================= */

    /* =========================================================
    |                                                          |
    |   Load the basis for system modules from the CORE folder |
    |                                                          |
    ========================================================= */

    mimimiInclude('Module.php');

    /* =========================================================
    |                                                          |
    |   Declare the child class                                |
    |                                                          |
    ========================================================= */

    class MimimiVisitor extends MimimiModule {

        /* -------------------------------------
        |                                      |
        |   The last result of the RUN method  |
        |                                      |
        ------------------------------------- */

        protected $lastData = null;

        /* =====================================================
        |                                                      |
        |   Perform some actions required to fetch the         |
        |   current visitor's data                             |
        |                                                      |
        |   ------------------------------------------------   |
        |                                                      |
        |   Note: We store the fetched data in the visitor's   |
        |         session to prevent lags due broken DNS.      |
        |                                                      |
        |   ------------------------------------------------   |
        |                                                      |
        |   Input parameters:                                  |
        |       $params = MIXED = some parameters if you need  |
        |                                                      |
        |   ------------------------------------------------   |
        |                                                      |
        |   Output parameters:                                 |
        |       ARRAY = visitor's data                         |
        |                                                      |
        ===================================================== */

        public function run ( $params = '' ) {
            if (is_null($this->lastData)) {
                $data = NULL;
                $field = 'moduleVisitor-lastData';

                /* -----------------------------
                |                              |
                |   Maybe fetched data exists  |
                |   in the visitor's session?  |
                |                              |
                ----------------------------- */

                if ($this->cms->has->session) {
                    $data = $this->cms->session->get(
                        $field,
                        NULL
                    );
                }

                /* -----------------------------
                |                              |
                |   Otherwise collect a data   |
                |                              |
                ----------------------------- */

                if (!is_array($data)) {
                    $data = array(
                        'ip'    => mimimiServer('REMOTE_ADDR'),
                        'host'  => mimimiServer('REMOTE_HOST'),
                        'agent' => mimimiServer('HTTP_USER_AGENT')
                    );

                    /* -------------------------
                    |                          |
                    |   Use first IP only      |
                    |                          |
                    ------------------------- */

                    $proxyPath = explode(
                        ',',
                        $data['ip']
                    );
                    $nearestIP = trim($proxyPath[0]);

                    /* -------------------------
                    |                          |
                    |   Refine the host by IP  |
                    |                          |
                    ------------------------- */

                    if (empty($data['host'])) {
                        $hasDNS = function_exists('dns_get_record');
                        $hasIP6 = strpos($nearestIP, ':') !== FALSE;
                        $hasIP4 = strpos($nearestIP, '.') !== FALSE;

                        /* ---------------------
                        |                      |
                        |   Over DNS Resource  |
                        |   Records for IPv6   |
                        |                      |
                        --------------------- */

                        if ($hasDNS && $hasIP6) {
                            $address = @ inet_pton($nearestIP);
                            $address = @ unpack(
                                'H*hex',
                                $address
                            );
                            if (isset($address['hex'])) {
                                $reversed = implode(
                                    '.',
                                    array_reverse(
                                        str_split(
                                            $address['hex']
                                        )
                                    )
                                );
                                $record = @ dns_get_record(
                                    $reversed . '.ip6.arpa',
                                    DNS_PTR
                                );
                                $field = 'target';
                                $data['host'] = empty($record[0][$field])
                                                ? ''
                                                : trim($record[0][$field]);
                            }

                        /* ---------------------
                        |                      |
                        |   Over DNS Resource  |
                        |   Records for IPv4   |
                        |                      |
                        --------------------- */

                        } else if ($hasDNS && $hasIP4) {
                            $reversed = implode(
                                '.',
                                array_reverse(
                                    explode('.', $nearestIP)
                                )
                            );
                            $record = @ dns_get_record(
                                $reversed . '.in-addr.arpa',
                                DNS_PTR
                            );
                            $field = 'target';
                            $data['host'] = empty($record[0][$field])
                                            ? ''
                                            : trim($record[0][$field]);

                        /* ---------------------
                        |                      |
                        |   Over SERVER's      |
                        |   Reverse Lookup     |
                        |                      |
                        --------------------- */

                        } else {
                            $data['host'] = @ gethostbyaddr(
                                $nearestIP
                            );
                            if (!is_string($data['host'])) {
                                $data['host'] = '';
                            }
                        }
                    }

                    /* -------------------------
                    |                          |
                    |   Refined host cannot    |
                    |   be its IP              |
                    |                          |
                    ------------------------- */

                    if ($data['host'] == $data['ip']
                    || $data['host'] == $nearestIP) {
                        $data['host'] = '';
                    }

                    /* -------------------------
                    |                          |
                    |   Save this data to      |
                    |   the visitor's session  |
                    |                          |
                    ------------------------- */

                    if ($this->cms->has->session) {
                        $this->cms->session->set(
                            $field,
                            $data,
                            24 * 60 * 60
                        );
                    }
                }

                /* -----------------------------
                |                              |
                |   Remember for the future    |
                |                              |
                ----------------------------- */

                $this->lastData = $data;
            }

            /* ---------------------------------
            |                                  |
            |   Return the data                |
            |                                  |
            --------------------------------- */

            return $this->lastData;
        }
    };

    /* =========================================================
    |                                                          |
    |   Block trailing injections                              |
    |                                                          |
    ========================================================= */

    return;