<?php
/**
 * -------------------------------------------------------------------------
 *
 * The module for working with the "settings" database table.
 *
 * -------------------------------------------------------------------------
 *
 * Overridden properties below are:
 *     PROTECTED  $table  ───────────> PUBLIC
 *     PROTECTED  $tableFields
 *                      ├──> id     INTEGER (auto incremented)
 *                      ├──> alias  STRING  (length 64)
 *                      ├──> type   STRING  (length 16)
 *                      └──> value  STRING  (length 4096)
 *     PROTECTED  $tableKeys
 *                      ├──> id     UNIQUE
 *                      └──> alias  UNIQUE
 *     PROTECTED  $demoRows
 *                      └──> mode_production  BOOLEAN
 *
 * Implemented properties below are:
 *     PROTECTED  $memory  -->  to cache settings entries until the app terminates
 *
 * Implemented methods below are:
 *     getItem                               ( $alias )  -->  to read a setting by its alias
 *        └──> getItems                      (        )  -->  to read all settings at once
 *                └──> PRIVATE  indexArray   ( $list  )  -->  to index settings by their aliases
 *     branchArray                           ( $list  )  -->  to branch settings to display them as nodes
 *     decodeItem                            ( $entry )  -->  to decode a setting value according to its type
 *           ├──> PROTECTED  decodeAsBoolean ( $entry )
 *           ├──> PROTECTED  decodeAsInteger ( $entry )
 *           ├──> PROTECTED  decodeAsString  ( $entry )
 *           ├──> PROTECTED  decodeAsText    ( $entry )
 *           ├──> PROTECTED  decodeAsHtml    ( $entry )
 *           └──> PROTECTED  decodeAsArray   ( $entry )
 *
 * -------------------------------------------------------------------------
 *
 * @package    MimimiFramework
 * @subpackage Modules
 * @copyright  2022 MiMiMi Community
 *             https://mimimi.software/
 * @license    GPL-2.0
 *             https://opensource.org/license/gpl-2-0/
 *
 * -------------------------------------------------------------------------
 */

    mimimiInclude ( 'ModuleWithTable.php' );
    class MimimiSettings extends MimimiModuleWithTable {

        /**
         * -----------------------------------------------------------------
         *
         * Specify a name of the database table to store settings.
         *
         * -----------------------------------------------------------------
         *
         * @access public
         * @var    string
         *
         * -----------------------------------------------------------------
         */

        public $table = 'settings';

        /**
         * -----------------------------------------------------------------
         *
         * Define a database table structure.
         *
         * -----------------------------------------------------------------
         *
         * @access protected
         * @var    array
         *
         * -----------------------------------------------------------------
         */

        protected $tableFields = [
                      '`id`     BIGINT(20)     NOT NULL  AUTO_INCREMENT    COMMENT "setting system identifier"',
                      '`alias`  VARCHAR(64)    NOT NULL                    COMMENT "alias to look up this setting"',
                      '`type`   VARCHAR(16)    NOT NULL  DEFAULT "string"  COMMENT "type of stored value"',
                      '`value`  VARCHAR(4096)  NOT NULL                    COMMENT "setting value"',
                      '`hint`   VARCHAR(512)   NOT NULL                    COMMENT "hint about this setting"'
                  ];

        /**
         * -----------------------------------------------------------------
         *
         * Define a list of table keys to speed up the database operations
         * related to settings.
         *
         * -----------------------------------------------------------------
         *
         * @access protected
         * @var    array
         *
         * -----------------------------------------------------------------
         */

        protected $tableKeys = [
                      'PRIMARY KEY ( `id`    )',
                      'UNIQUE  KEY ( `alias` )'
                  ];

        /**
         * -----------------------------------------------------------------
         *
         * Specify demo rows that will be used as default setting entries
         * if the database does not have a table named "settings". In this
         * case, all demo rows will be automatically added to the newly
         * created table.
         *
         * -----------------------------------------------------------------
         *
         * @access protected
         * @var    array
         *
         * -----------------------------------------------------------------
         */

        protected $demoRows = [
                      [ 'alias' => 'mode_production', 'type' => 'boolean', 'value' => '0', 'hint' => 'ZERO if your site is currently in demo mode, ONE if it is in production mode.' ]
                  ];

        /**
         * -----------------------------------------------------------------
         *
         * Define an area to cache all settings at once. This helps reduce
         * the number of database calls when the page renderer requests
         * settings one at a time.
         *
         * -----------------------------------------------------------------
         *
         * @access protected
         * @var    array
         *
         * -----------------------------------------------------------------
         */

        protected $memory = NULL;

        /**
         * -----------------------------------------------------------------
         *
         * Indexes an array of settings by their aliases.
         *
         * -----------------------------------------------------------------
         *
         * @private
         * @param   array       $list  The array to be indexable.
         * @return  array|bool         ARRAY on success.
         *                             FALSE on failure.
         *
         * -----------------------------------------------------------------
         */

        private function indexArray ( $list ) {
            $result = FALSE;
            if ( is_array ( $list ) ) {
                $next   = 0;
                $result = [ ];
                foreach ( $list as $row ) {
                    $index = isset ( $row[ 'alias' ] )
                                   ? $row[ 'alias' ]
                                   : $next++;
                    $result[ $index ] = $row;
                }
            }
            return $result;
        }

        /**
         * -----------------------------------------------------------------
         *
         * Branches an array of settings according to the words of the alias.
         *
         * -----------------------------------------------------------------
         *
         * @private
         * @param   array       $list  The array that needs to be branched.
         * @return  array|bool         ARRAY on success.
         *                             FALSE on failure.
         *
         * -----------------------------------------------------------------
         */

        public function branchArray ( $list ) {
            $result = FALSE;
            if ( is_array ( $list ) ) {
                $result = [ ];
                foreach ( $list as $row ) {
                    if ( isset ( $row[ 'alias' ] ) ) {
                        $ptr = & $result;
                        $nodes = preg_split ( '~_+~u', $row[ 'alias' ] );
                        do {
                            $node = array_shift ( $nodes );
                            if ( $nodes ) {
                                if ( ! isset ( $ptr[ $node ] ) ) {
                                    $ptr[ $node ] = [ ];
                                }
                                $ptr = & $ptr[ $node ];
                            } else {
                                $ptr[ $node ] = $row;
                            }
                        } while ( $nodes );
                    }
                }
            }
            return $result;
        }

        /**
         * -----------------------------------------------------------------
         *
         * Retrieves all settings entries.
         *
         * -----------------------------------------------------------------
         *
         * These entries are sorted alphabetically by their aliases. Please
         * note that "t1" below is an alias for the "settings" database
         * table.
         *
         * -----------------------------------------------------------------
         *
         * @public
         * @return  array|bool  ARRAY on success. Each element is an array, like a row obtained from a database table.
         *                      FALSE on failure. This means no entries were found.
         *
         * -----------------------------------------------------------------
         */

        public function getItems ( ) {
            if ( is_null ( $this->memory ) ) {
                $sorter = [
                    'orderby' => [ 't1.alias' => 'asc' ]
                ];
                $list         = $this->select     ( $sorter, 0, 1000000000 );
                $this->memory = $this->indexArray ( $list                  );
            }
            return $this->memory;
        }

        /**
         * -----------------------------------------------------------------
         *
         * Retrieves a setting entry by its alias.
         *
         * -----------------------------------------------------------------
         *
         * @public
         * @param   string      $alias  The alias of the setting you are looking for.
         * @return  array|bool          ARRAY on success. It contains a row obtained from a database table.
         *                              FALSE on failure. This means the entry you are looking for was not found.
         *
         * -----------------------------------------------------------------
         */

        public function getItem ( $alias ) {
            $entries = $this->getItems ( );
            return isset ( $entries[ $alias ] )
                         ? $entries[ $alias ]
                         : FALSE;
        }

        /**
         * -----------------------------------------------------------------
         *
         * Decodes a setting entry according to its type.
         *
         * -----------------------------------------------------------------
         *
         * Please note that the decoded value is always stored under the
         * index "decoded".
         *
         * -----------------------------------------------------------------
         *
         * @public
         * @param   array  $entry  The entry that needs to be decoded.
         * @return  array          Decoded entry.
         *
         * -----------------------------------------------------------------
         */

        public function decodeItem ( $entry ) {
            $type = isset (              $entry[ 'type' ] )
                          ? strtolower ( $entry[ 'type' ] )
                          : '';
            switch ( $type ) {
                case 'bool':
                case 'boolean': $value = $this->decodeAsBoolean ( $entry ); break;
                case 'int':
                case 'integer':
                case 'number':  $value = $this->decodeAsInteger ( $entry ); break;
                case 'string':  $value = $this->decodeAsString  ( $entry ); break;
                case 'text':    $value = $this->decodeAsText    ( $entry ); break;
                case 'html':    $value = $this->decodeAsHtml    ( $entry ); break;
                case 'array':
                case 'list':    $value = $this->decodeAsArray   ( $entry ); break;
                default:        $value = isset ( $entry[ 'value' ] )
                                               ? $entry[ 'value' ]
                                               : '';
            }
            $entry[ 'decoded' ] = $value;
            return $entry;
        }

        /**
         * -----------------------------------------------------------------
         *
         * Decodes a setting value according to Boolean type.
         *
         * -----------------------------------------------------------------
         *
         * @protected
         * @param   array  $entry  The entry that needs to be decoded.
         * @return  bool           Decoded value.
         *
         * -----------------------------------------------------------------
         */

        protected function decodeAsBoolean ( $entry ) {
            return ! empty ( $entry[ 'value' ] );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Decodes a setting value according to Integer type.
         *
         * -----------------------------------------------------------------
         *
         * @protected
         * @param   array  $entry  The entry that needs to be decoded.
         * @return  int            Decoded value.
         *
         * -----------------------------------------------------------------
         */

        protected function decodeAsInteger ( $entry ) {
            return isset (            $entry[ 'value' ] )
                         ? @ intval ( $entry[ 'value' ] )
                         : 0;
        }

        /**
         * -----------------------------------------------------------------
         *
         * Decodes a setting value according to String type.
         *
         * -----------------------------------------------------------------
         *
         * @protected
         * @param   array   $entry  The entry that needs to be decoded.
         * @return  string          Decoded value.
         *
         * -----------------------------------------------------------------
         */

        protected function decodeAsString ( $entry ) {
            $spaces = '~\s+~u'         ;
            $trails = '~(^\s+|\s+)$~u' ;
            return  ! isset (                    $entry[ 'value' ] )
                            ? ''
                            : preg_replace (     $spaces, ' ',
                                  preg_replace ( $trails, '',
                                                 $entry[ 'value' ]
                                  )
                              );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Decodes a setting value according to Text type.
         *
         * -----------------------------------------------------------------
         *
         * @protected
         * @param   array   $entry  The entry that needs to be decoded.
         * @return  string          Decoded value.
         *
         * -----------------------------------------------------------------
         */

        protected function decodeAsText ( $entry ) {
            $spaces = '~[ \t]+~u'          ;
            $CRLFs  = '~[ \t]*[\r\n]\s*~u' ;
            $trails = '~(^\s+|\s+)$~u'     ;
            return  ! isset (                        $entry[ 'value' ] )
                            ? ''
                            : preg_replace (         $spaces, ' ',
                                  preg_replace (     $CRLFs,  "\r\n\r\n",
                                      preg_replace ( $trails, '',
                                                     $entry[ 'value' ]
                                      )
                                  )
                              );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Decodes a setting value according to Html type.
         *
         * -----------------------------------------------------------------
         *
         * @protected
         * @param   array   $entry  The entry that needs to be decoded.
         * @return  string          Decoded value.
         *
         * -----------------------------------------------------------------
         */

        protected function decodeAsHtml ( $entry ) {
            $CRLFs    = '~[ \t]*[\r\n]+([ \t]+[\r\n]+)*~u'   ;
            $tags     = '~<(/?[a-z][a-z0-9]*)[\s/][^>]*>~ui' ;
            $trails   = '~(^\s+|\s+)$~u'                     ;
            $bads     = '~(<!--|<\?).*$~us'                  ;
            $comments = '~<!--.*?-->~us'                     ;
            $codes    = '~<\?.*?\?>~us'                      ;
            return  ! isset (                                    $entry[ 'value' ] )
                            ? ''
                            : preg_replace (                     $CRLFs,    "\r\n",
                                  preg_replace (                 $tags,     '<$1>',
                                      preg_replace (             $trails,   '',
                                          preg_replace (         $bads,     '',
                                              preg_replace (     $comments, '',
                                                  preg_replace ( $codes,    '',
                                                                 $entry[ 'value' ]
                                                  )
                                              )
                                          )
                                      )
                                  )
                              );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Decodes a setting value according to Array type.
         *
         * -----------------------------------------------------------------
         *
         * @protected
         * @param   array   $entry  The entry that needs to be decoded.
         * @return  array           Decoded value.
         *
         * -----------------------------------------------------------------
         */

        protected function decodeAsArray ( $entry ) {
            $commas = '~[\s,]~u'             ;
            $trails = '~(^[\s,]+|[\s,]+$)~u' ;
            return  ! isset (                    $entry[ 'value' ] )
                            ? [ ]
                            : preg_split (       $commas,
                                  preg_replace ( $trails, '',
                                                 $entry[ 'value' ]
                                  )
                              );
        }
    };
