<?php
/**
 * -------------------------------------------------------------------------
 *
 * Module to request pages attached to the topmenu.
 *
 * -------------------------------------------------------------------------
 *
 * To append a new item to the topmenu, you need place a directory with
 * a PHP file the same name. For example, the Help directory and its
 * Help.php file. This file will become a child module that can be accessed
 * via a URL, like this one:
 *
 *     https://your.site/menu/help
 *
 * You can also create a new submenu by nesting a module within that child
 * module. For example, the About directory and its About.php file. This
 * file will be accessed via a URL like this:
 *
 *     https://your.site/menu/help/about
 *
 * The same URL may have a second format with arbitrary parameters if the
 * first parameter does not overlap the name of any nested module (remember,
 * for example, the "about" mentioned above):
 *
 *     https://your.site/menu/help/PARAMETER-1/PARAMETER-2/AND-SO-ON
 *
 * You may specify the submenu items you want to display on the screen by
 * declaring the "getMapItems" method in the module of the desired submenu.
 * This method must return an array of menu elements. Each element is an
 * indexed array, like this:
 *
 *     [   'name'     => 'About',
 *         'shortcut' => 'f1',
 *         'inactive' => TRUE,
 *         'icon'     => 'Themes/default/images/icon-about.svg',
 *         'toolicon' => '<svg xmlns="..."><path d="..."/></svg>',
 *         'url'      => 'menu/help/about',
 *         'title'    => 'Opens information about application components.'
 *     ]
 *
 *     NOTE: All of these indexes are optional.
 *
 * Typically, the method returns an array of one element. But it can also
 * return a series of elements if that submenu item is actually consists
 * of multiple elements.
 *
 * The method may use the "subitems" index in any item to specify its
 * submenu if you need to. For example:
 *
 *     [   'name'     => 'About',
 *         'icon'     => 'Themes/default/images/icon-about.svg',
 *         'url'      => 'menu/help/about',
 *         'title'    => 'Shows general information.',
 *         'subitems' => [   
 *                           [   'name'  => 'Application',
 *                               'url'   => 'menu/help/about/app',
 *                               'title' => 'Opens information about the application.'
 *                           ], [
 *                               'name'  => 'Components',
 *                               'url'   => 'menu/help/about/components',
 *                               'title' => 'Opens information about components.'
 *                           ]
 *                       ]
 *     ]
 *
 * Additionally, the "url" index can be empty to indicate a non-clickable
 * item. Or it can contain a Javascript instruction specifying an item that
 * causes the browser's javascript code to be executed.
 *
 * -------------------------------------------------------------------------
 *
 * @package    MimimiFramework
 * @subpackage Examples / IDE skeleton
 * @license    GPL-2.0
 *             https://opensource.org/license/gpl-2-0/
 * @copyright  2022 MiMiMi Community
 *             https://mimimi.software/
 *
 * -------------------------------------------------------------------------
 */

    mimimiInclude ( 'Collection.php', FALSE );

    class MyMimimiMenu extends MyMimimiCollection {

        /**
         * -----------------------------------------------------------------
         *
         * Renders a page associated with the clicked URL.
         *
         * -----------------------------------------------------------------
         *
         * @public
         * @param   string  $url  The URL that was clicked via the topmenu.
         * @return  bool          TRUE  if the page was rendered successfully.
         *                        FALSE if there is no associated module or operation.
         *
         * -----------------------------------------------------------------
         */

        public function run ( $url = '' ) {
            $module = $url ? $this->findModule ( $url, 'run' )
                           : FALSE;
            return $module ? $module->run ( $url )
                           : FALSE;
        }

        /**
         * -----------------------------------------------------------------
         *
         * Gets a map of pages attached to the menu referenced by its URL.
         *
         * -----------------------------------------------------------------
         *
         * @public
         * @param   string  $url  EMPTY STRING if you want to get a map for the topmenu.
         *                        STRING       if you need a map for the submenu referenced by the URL.
         * @return  array         Array of found map entries.
         *
         * -----------------------------------------------------------------
         */

        public function getMap ( $url = '' ) {
            $source = $url;
            $module = $url ? $this->findModule ( $url )
                           : $this;
            $baseUrl = 'menu/' . ( $module === $this
                                             ? $source
                                             : substr ( $source, 0, strlen ( $source ) - strlen ( $url ) ) );
            return $module ? $this->getMapFor ( $module, $baseUrl )
                           : [ ];
        }

        /**
         * -----------------------------------------------------------------
         *
         * Reset the namespace simulator. You should reset it because this
         * module may have child or more nested modules with functionality
         * related to their hierarchy.
         *
         * -----------------------------------------------------------------
         *
         * @var    string
         * @access protected
         *
         * -----------------------------------------------------------------
         */

        protected $myNodeFile = __FILE__;
    };
