<?php
/**
 * -------------------------------------------------------------------------
 *
 * Module to provide specific routines for site templates.
 *
 * -------------------------------------------------------------------------
 *
 * @package    MimimiFramework
 * @subpackage Examples / Static Pages only
 * @license    GPL-2.0
 *             https://opensource.org/license/gpl-2-0/
 * @copyright  2022 MiMiMi Community
 *             https://mimimi.software/
 *
 * -------------------------------------------------------------------------
 */

    mimimiInclude ( 'Helper/Helper.php' );

    class MyMimimiHelper extends MimimiHelper {

        /**
         * -----------------------------------------------------------------
         *
         * Checks if a template file exists.
         *
         * -----------------------------------------------------------------
         *
         * Please note that the template file name must be specified relative
         * to the "static.pages.only/Themes/default" directory.
         *
         * -----------------------------------------------------------------
         *
         * @public
         * @param   string  $template  (optional) The relative name of the template file to be checked.
         * @return  bool               FALSE if this file does not exist.
         *
         * -----------------------------------------------------------------
         */

        public function run ( $template = '' ) {
            if ( $template ) {
                $path = helperGetThemePath ( );
                $file = mimimiBasePath ( $path . $template, TRUE );
                return file_exists ( $file )
                    && is_file     ( $file )
                    && is_readable ( $file );
            }
            return FALSE;
        }
    };

    /**
     * ---------------------------------------------------------------------
     *
     * If you want to have specific routines (imaginary "tags") for business
     * logic in template files, you can implement them below. It is recommended
     * to add a prefix "helper..." to the name of such routines to make it
     * easier to understand the origin of these "tags" when using them.
     *
     * ---------------------------------------------------------------------
     */

        /**
         * -----------------------------------------------------------------
         *
         * Routine to return the relative path to the current site theme.
         *
         * -----------------------------------------------------------------
         */

        function helperGetThemePath ( ) {
            return MIMIMI_APP_FOLDER . 'Themes/' . MIMIMI_APP_THEME . '/';
        }

        /**
         * -----------------------------------------------------------------
         *
         * Routine to check if the site is currently running in demo mode.
         *
         * -----------------------------------------------------------------
         */

        function helperIsDemo ( ) {
            return ! empty ( STATICPAGESONLY_DEMO_MODE );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Routine to check if the current visitor is an admin.
         *
         * -----------------------------------------------------------------
         */

        function helperAmIAdmin ( ) {
            global $app;
                   return $app->amIAdmin ( );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Routine to set a new timeout period.
         *
         * -----------------------------------------------------------------
         */

        function helperSetTimeout ( $type, $pause ) {
            global $app;
                   $app->setTimeout ( $type, $pause );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Routine to log out the current administrator.
         *
         * -----------------------------------------------------------------
         */

        function helperLogoutAdmin ( ) {
            global $app;
                   $app->setAdminLabel ( '' );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Routine to log in the current visitor.
         *
         * -----------------------------------------------------------------
         */

        function helperLoginAdmin ( ) {
            global $app;
                   $my = $app->getMyLabel    (     );
                         $app->setAdminLabel ( $my );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Routine to check if the security timeout has expired.
         *
         * -----------------------------------------------------------------
         */

        function helperCheckForTimeout ( $type ) {
            global $app;
                   return $app->getTimeout ( $type ) <= time ( );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Routine to check if the login entered by a visitor is valid.
         *
         * -----------------------------------------------------------------
         */

        function helperValidateLogin ( $login ) {
            return $login == STATICPAGESONLY_ADMIN_LOGIN;
        }

        /**
         * -----------------------------------------------------------------
         *
         * Routine to check if the password entered by a visitor is valid.
         *
         * -----------------------------------------------------------------
         */

        function helperValidatePassword ( $login, $pass ) {
            $hash = helperComputeHash ( $pass );
            return $hash == STATICPAGESONLY_ADMIN_PASSWORD_HASH;
        }

        /**
         * -----------------------------------------------------------------
         *
         * Routine to check if the CSS File URL entered by an admin is valid.
         *
         * -----------------------------------------------------------------
         */

        function helperValidateCssUrl ( $url ) {
            return helperGetCssUrl ( $url ) == $url
                && preg_match ( '~[^/\\\\]\.css$~ui', $url );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Routine to check if the CSS File was uploaded successfully.
         *
         * -----------------------------------------------------------------
         */

        function helperValidateCssFile ( $data ) {
            return isset ( $data[ 'error' ] )
                && $data[ 'error' ] == UPLOAD_ERR_OK
                && preg_match ( '~[^/\\\\]\.css$~ui', $data[ 'name' ] );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Routine to check if the static page URL entered by an admin is valid.
         *
         * -----------------------------------------------------------------
         */

        function helperValidatePageUrl ( $url ) {
            return helperGetPageUrl ( $url ) == $url
                && preg_match ( '~(?<!\.css|\.js|\.xml|\.tpl|\.php|\.php\d)$~ui', $url );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Routine to check if the TPL file was uploaded successfully.
         *
         * -----------------------------------------------------------------
         */

        function helperValidatePageFile ( $data ) {
            return isset ( $data[ 'error' ] )
                && $data[ 'error' ] == UPLOAD_ERR_OK
                && preg_match ( '~[^/\\\\]\.(tpl|htm|html)$~ui', $data[ 'name' ] );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Routine to check if the bbCode File URL entered by an admin is valid.
         *
         * -----------------------------------------------------------------
         */

        function helperValidateBBcodeUrl ( $url ) {
            return helperGetBBcodeUrl ( $url ) == $url
                && preg_match ( '~[^/\\\\]\.tpl$~ui', $url );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Routine to check if the bbCode File was uploaded successfully.
         *
         * -----------------------------------------------------------------
         */

        function helperValidateBBcodeFile ( $data ) {
            return isset ( $data[ 'error' ] )
                && $data[ 'error' ] == UPLOAD_ERR_OK
                && preg_match ( '~[^/\\\\]\.(tpl|htm|html)$~ui', $data[ 'name' ] );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Routine to check if the Fragment File URL entered by an admin is valid.
         *
         * -----------------------------------------------------------------
         */

        function helperValidateFragmentUrl ( $url ) {
            return helperGetFragmentUrl ( $url ) == $url
                && preg_match ( '~[^/\\\\]\.tpl$~ui', $url );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Routine to check if the Fragment File was uploaded successfully.
         *
         * -----------------------------------------------------------------
         */

        function helperValidateFragmentFile ( $data ) {
            return isset ( $data[ 'error' ] )
                && $data[ 'error' ] == UPLOAD_ERR_OK
                && preg_match ( '~[^/\\\\]\.(tpl|htm|html)$~ui', $data[ 'name' ] );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Routine to compute a password hash.
         *
         * -----------------------------------------------------------------
         */

        function helperComputeHash ( $pass ) {
            global $app;
                   return $app->computeHash ( $pass );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Routine to convert a flat array of URLs into a hierarchical one.
         *
         * -----------------------------------------------------------------
         */

        function helperConvertUrlList ( $flat ) {
            $hierarchical = [ ];
            foreach ( $flat as $url ) {
                $ptr = & $hierarchical;
                $segments = preg_split ( '~/+~u', $url );
                while ( $segments ) {
                    $index = array_shift ( $segments );
                    if ( isset ( $ptr[ $index ] ) ) {
                        if ( $segments ) {
                            if ( is_string ( $ptr[ $index ] ) ) {
                                    $ptr[ $index ] = [ '' => $ptr[ $index ] ];
                            }
                        } else {
                            if ( is_array ( $ptr[ $index ] ) ) {
                                if ( ! isset ( $ptr[ $index ][ '' ] ) ) {
                                    $ptr[ $index ][ '' ] = $url;
                                }
                            }
                        }
                    } else {
                        if ( $segments ) {
                            $ptr[ $index ] = [ ];
                        } else {
                            $ptr[ $index ] = $url;
                        }
                    }
                    $ptr = & $ptr[ $index ];
                }
            }
            return $hierarchical;
        }

        /**
         * -----------------------------------------------------------------
         *
         * Routine to collect existing styles as a flat or hierarchical array.
         *
         * -----------------------------------------------------------------
         */

        function helperCollectStyles ( $asFlat = FALSE ) {
            global $app;
                   $urls = $app->database->styles->getMap ( FALSE );
                   return $asFlat ? $urls
                                  : helperConvertUrlList ( $urls );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Routine to collect existing pages as a flat or hierarchical array.
         *
         * -----------------------------------------------------------------
         */

        function helperCollectPages ( $asFlat = FALSE ) {
            global $app;
                   $urls = $app->database->pages->getMap ( );
                   return $asFlat ? $urls
                                  : helperConvertUrlList ( $urls );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Routine to collect existing bbCodes as a flat or hierarchical array.
         *
         * -----------------------------------------------------------------
         */

        function helperCollectBBcodes ( $asFlat = FALSE ) {
            global $app;
                   $urls = $app->database->bbcodes->getMap ( FALSE );
                   return $asFlat ? $urls
                                  : helperConvertUrlList ( $urls );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Routine to collect existing Fragments as a flat or hierarchical array.
         *
         * -----------------------------------------------------------------
         */

        function helperCollectFragments ( $asFlat = FALSE ) {
            global $app;
                   $urls = $app->database->fragments->getMap ( FALSE );
                   return $asFlat ? $urls
                                  : helperConvertUrlList ( $urls );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Routine to update a setting value.
         *
         * -----------------------------------------------------------------
         */

        function helperUpdateSetting ( $name, $value ) {
            $root = mimimiBasePath ( MIMIMI_APP_FOLDER );
            $file = $root . '/Constants.php';
            $data = file_get_contents ( $file );
            if ( is_string ( $data ) ) {
                $pattern = '~^' .
                               '(' .
                                   '.*?[\s/;,{}]' .
                                   'define\s*\(' .
                                       '\s*' .
                                       '(["\'])' . preg_quote ( $name, '~' ) . '\2' .
                                       '\s*' .
                                    ',' .
                               ')' .
                                       '\s*' .
                                       '([a-z0-9.]+|(["\'])[^\r\n]*\4)' .
                                       '\s*' .
                               '(' .
                                   '\);[ \t]*[\r\n]' .
                                   '.*?' .
                               ')' .
                           '$~uis';
                if ( preg_match ( $pattern, $data ) ) {
                    if ( is_bool ( $value ) ) {
                        $value = $value ? ' TRUE '
                                        : ' FALSE ';
                    } else if ( is_string ( $value ) ) {
                        $value = preg_replace ( '~(["$\\\\])~u',    '\\$1', $value );
                        $value = preg_replace ( "~\r~u",            '\r',   $value );
                        $value = preg_replace ( "~\n~u",            '\n',   $value );
                        $value = preg_replace ( "~\t~u",            '\t',   $value );
                        $value = preg_replace ( '~[\s\x00-\x20]~u', ' ',    $value );
                        $value = ' "' . $value . '" ';
                    } else {
                        $value = ' ' . intval ( $value ) . ' ';
                    }
                    $begin = preg_replace ( $pattern, '$1', $data );
                    $end   = preg_replace ( $pattern, '$5', $data );
                    $new   = rtrim ( $begin ) . $value . $end;
                    if ( $new != $data ) {
                        @ file_put_contents ( $file, $new );
                    }
                }
            }
        }

        /**
         * -----------------------------------------------------------------
         *
         * Routine to remove server-side includes from HTML markup.
         *
         * -----------------------------------------------------------------
         */

        function helperRemoveSSI ( $html, $andPhp = TRUE ) {
            $html = preg_replace ( '~<(%).*?\1>~us',            '', $html );
            $html = preg_replace ( '~<(%).*?$~us',              '', $html );
            $html = preg_replace ( '~<(\?)(?!php\s).*?\1>~uis', '', $html );
            $html = preg_replace ( '~<(\?)(?!php\s).*?$~uis',   '', $html );
            $html = preg_replace ( '~<!--#.*?-->~us',           '', $html );
            $html = preg_replace ( '~<!--#.*$~us',              '', $html );
            if ( ! $andPhp ) return $html;
            $html = preg_replace ( '~<(\?)php\s.*?\1>~uis',     '', $html );
            return  preg_replace ( '~<(\?)php\s.*?$~uis',       '', $html );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Routine to load content of the file located in the JS directory.
         *
         * -----------------------------------------------------------------
         */

        function helperLoadJS ( $filename ) {
            $file = helperGetThemePath ( ) . 'js/' . $filename;
            $file = mimimiBasePath ( $file, TRUE );
            $data = @ file_get_contents ( $file );
            return is_string ( $data ) ? $data
                                       : '';
        }

        /**
         * -----------------------------------------------------------------
         *
         * Routine to save content to the file located in the JS directory.
         *
         * -----------------------------------------------------------------
         */

        function helperSaveJS ( $filename, $data ) {
            $file = helperGetThemePath ( ) . 'js/' . $filename;
            $file = mimimiBasePath ( $file, TRUE );
            @ file_put_contents ( $file, $data );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Routine to return the relative path to the CSS files storage.
         *
         * -----------------------------------------------------------------
         */

        function helperGetCssPath ( ) {
            global $app;
                   return $app->database->styles->getStoragePath ( );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Routine to delete a CSS file located in storage at a URL.
         *
         * -----------------------------------------------------------------
         */

        function helperDeleteCss ( $url ) {
            global $app;
                   return $app->database->styles->remove ( $url );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Routine to load CSS from a style file located in storage at a URL.
         *
         * -----------------------------------------------------------------
         */

        function helperLoadCss ( $url ) {
            global $app;
                   return $app->database->styles->run ( $url );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Routine to save content to a style file located in storage at a URL.
         *
         * -----------------------------------------------------------------
         */

        function helperSaveCss ( $url, $content ) {
            global $app;
                   return $app->database->styles->save ( $url, $content );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Routine to upload a CSS file in storage at a URL.
         *
         * -----------------------------------------------------------------
         */

        function helperUploadCss ( $url, $data ) {
            global $app;
                   return $app->database->styles->upload ( $url, $data );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Routine to return the relative path to the pages storage.
         *
         * -----------------------------------------------------------------
         */

        function helperGetPagesPath ( ) {
            global $app;
                   return $app->database->pages->getStoragePath ( );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Routine to delete a page TPL file located in storage at a URL.
         *
         * -----------------------------------------------------------------
         */

        function helperDeletePage ( $url ) {
            global $app;
                   return $app->database->pages->remove ( $url );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Routine to load HTML from a page TPL file located in storage at a URL.
         *
         * -----------------------------------------------------------------
         */

        function helperLoadPage ( $url ) {
            global $app;
                   return $app->database->pages->run ( $url );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Routine to save content to a page TPL file located in storage at a URL.
         *
         * -----------------------------------------------------------------
         */

        function helperSavePage ( $url, $content ) {
            global $app;
                   return $app->database->pages->save ( $url, $content );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Routine to upload a page TPL file in storage at a URL.
         *
         * -----------------------------------------------------------------
         */

        function helperUploadPage ( $url, $data ) {
            global $app;
                   return $app->database->pages->upload ( $url, $data );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Routine to delete a bbCode file located in storage at a URL.
         *
         * -----------------------------------------------------------------
         */

        function helperDeleteBBcode ( $url ) {
            global $app;
                   return $app->database->bbcodes->remove ( $url );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Routine to load bbCode from a TPL file located in storage at a URL.
         *
         * -----------------------------------------------------------------
         */

        function helperLoadBBcode ( $url ) {
            global $app;
                   return $app->database->bbcodes->run ( $url );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Routine to save content to a bbCode TPL file located in storage at a URL.
         *
         * -----------------------------------------------------------------
         */

        function helperSaveBBcode ( $url, $content ) {
            global $app;
                   return $app->database->bbcodes->save ( $url, $content );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Routine to upload a bbCode TPL file in storage at a URL.
         *
         * -----------------------------------------------------------------
         */

        function helperUploadBBcode ( $url, $data ) {
            global $app;
                   return $app->database->bbcodes->upload ( $url, $data );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Routine to return the relative path to the bbCode files storage.
         *
         * -----------------------------------------------------------------
         */

        function helperGetBBcodesPath ( ) {
            global $app;
                   return $app->database->bbcodes->getStoragePath ( );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Routine to delete a Fragment file located in storage at a URL.
         *
         * -----------------------------------------------------------------
         */

        function helperDeleteFragment ( $url ) {
            global $app;
                   return $app->database->fragments->remove ( $url );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Routine to load Fragment from a TPL file located in storage at a URL.
         *
         * -----------------------------------------------------------------
         */

        function helperLoadFragment ( $url ) {
            global $app;
                   return $app->database->fragments->run ( $url );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Routine to save content to a Fragment TPL file located in storage at a URL.
         *
         * -----------------------------------------------------------------
         */

        function helperSaveFragment ( $url, $content ) {
            global $app;
                   return $app->database->fragments->save ( $url, $content );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Routine to upload a Fragment TPL file in storage at a URL.
         *
         * -----------------------------------------------------------------
         */

        function helperUploadFragment ( $url, $data ) {
            global $app;
                   return $app->database->fragments->upload ( $url, $data );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Routine to return the relative path to the Fragment files storage.
         *
         * -----------------------------------------------------------------
         */

        function helperGetFragmentsPath ( ) {
            global $app;
                   return $app->database->fragments->getStoragePath ( );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Routine to format HTML.
         *
         * -----------------------------------------------------------------
         */

        function helperFormatHtml ( $html, $margin = '' ) {
            $crlf     = "\r\n";
            $first    = $margin;
            $result   = '';
            $breaks   = [ 'address',
                          'article',
                          'aside',
                          'audio',
                          'blockquote',
                          'body',
                          'canvas',
                          'datalist',
                          'details',
                          'dialog',
                          'dd',
                          'div',
                          'dl',
                          'dt',
                          'fieldset',
                          'figcaption',
                          'figure',
                          'footer',
                          'form',
                          'frame',
                          'frameset',
                          'h1',
                          'h2',
                          'h3',
                          'h4',
                          'h5',
                          'h6',
                          'head',
                          'header',
                          'hgroup',
                          'iframe',
                          'legend',
                          'li',
                          'main',
                          'map',
                          'menu',
                          'nav',
                          'noscript',
                          'object',
                          'ol',
                          'p',
                          'path',
                          'picture',
                          'pre',
                          'script',
                          'section',
                          'select',
                          'style',
                          'summary',
                          'svg',
                          'table',
                          'tbody',
                          'td',
                          'template',
                          'textarea',
                          'tfoot',
                          'th',
                          'thead',
                          'title',
                          'tr',
                          'ul',
                          'video' ];
            $onelines = [ '!doctype',
                          'base',
                          'br',
                          'hr',
                          'link',
                          'meta' ];
            $empties  = [ 'area',
                          'base',
                          'br',
                          'col',
                          'embed',
                          'hr',
                          'img',
                          'input',
                          'link',
                          'meta',
                          'param',
                          'source',
                          'track',
                          'wbr' ];
            $pattern  = '~^(.*?)(<(!doctype|(/?)([a-z][a-z0-9]*))([\s/][^>]*>|>))(.*)$~uis';
            $stack    = [ ];
            while ( preg_match ( $pattern, $html ) ) {
                $start  = preg_replace ( $pattern, '$1', $html );
                $tag    = preg_replace ( $pattern, '$2', $html );
                $closer = preg_replace ( $pattern, '$4', $html );
                $name   = strtolower ( $closer ? preg_replace ( $pattern, '$5', $html )
                                                : preg_replace ( $pattern, '$3', $html ) );
                if ( $closer ) {
                    $test = end ( $stack );
                    if ( $test == $name ) {
                        array_pop ( $stack );
                        if ( in_array ( $name, $breaks ) ) {
                            $margin = substr ( $margin, 0, -4 );
                            $result = preg_replace ( '~\s+$~u', '', $result . $start ) . $crlf .
                                      $margin . $tag . $crlf .
                                      $margin;
                        } else {
                            $result .= $start . $tag;
                        }
                    } else {
                        $result .= $start;
                    }
                } else {
                    if ( in_array ( $name, $onelines ) ) {
                        $result = preg_replace ( '~\s+$~u', '', $result . $start ) . $crlf .
                                  $margin . $tag . $crlf .
                                  $margin;
                    } else if ( in_array ( $name, $breaks ) ) {
                        $result = preg_replace ( '~\s+$~u', '', $result . $start ) . $crlf .
                                  $margin . $tag . $crlf;
                        if ( ! preg_match ( '~/>$~u', $tag ) ) {
                            $stack[ ] = $name;
                            $margin .= '    ';
                        }
                        $result .= $margin;
                    } else {
                        if ( ! in_array   ( $name, $empties )
                        &&   ! preg_match ( '~/>$~u', $tag  ) ) {
                            $stack[ ] = $name;
                        }
                        $result .= $start . $tag;
                    }
                }
                $html = preg_replace ( $pattern, '$7', $html );
            }
            $result .= $html;
            while ( $stack ) {
                $name = array_pop ( $stack );
                if ( in_array ( $name, $breaks ) ) {
                    $margin = substr ( $margin, 0, -4 );
                    $result = preg_replace ( '~\s+$~u', '', $result ) . $crlf .
                              $margin . '</' . $name . '>' . $crlf .
                              $margin;
                } else {
                    $result .= '</' . $name . '>';
                }
            }
            return $first . preg_replace ( '~\s+$~u', '', $result ) . $crlf;
        }

        /**
         * -----------------------------------------------------------------
         *
         * Routine to compile a file for saving into the pages storage.
         *
         * -----------------------------------------------------------------
         */

        function helperCompilePage ( $entry ) {
            $attrs = '(\s[^>]*?)?';
            $start = '~^' .
                         '.*?' .
                         '<body' . $attrs . '>' .
                             '\s*' .
                             '~uis';
            $end   =         '~\s*' .
                         '</body>' .
                         '.*' .
                     '$~uis';
            $body = preg_replace ( $start, '', $entry[ 'body' ] );
            if ( $body != $entry[ 'body' ] ) {
                $body = preg_replace ( $end, '', $body );
            }
            $crlf = "\r\n";
            return '<!DOCTYPE html>' . $crlf .
                   '<html lang="en">' . $crlf .
                   '    <head>' . $crlf .
                   '        <title>' . printValue ( $entry[ 'title' ], FALSE ) . '</title>' . $crlf .
                   '        <meta name="description" content="' . printValue ( $entry[ 'meta' ], FALSE ) . '">' . $crlf .
                   $crlf .
                   '        <!-- You can limit the list of allowed editing tools for this page to a specific name or any one (use an empty string) -->' . $crlf .
                   '        <meta name="editing-permission" content="' . printValue ( $entry[ 'permission' ], FALSE ) . '">' . $crlf .
                   $crlf .
                   '        <!-- Note that this style URL must be specified relative to the "static.pages.only/Themes/css/pages" directory -->' . $crlf .
                   '        <link rel="stylesheet" href="' . printValue ( $entry[ 'css' ], FALSE ) . '">' . $crlf .
                   '    </head>' . $crlf .
                   '    <body>' . $crlf .
                   helperFormatHtml ( $body, '        ' ) .
                   '    </body>' . $crlf .
                   '</html>' . $crlf;
        }

        /**
         * -----------------------------------------------------------------
         *
         * Routine to clean unwanted tags and attributes and format
         * a page's HTML source code.
         *
         * -----------------------------------------------------------------
         */

        function helperCleanBody ( $html ) {
            return $html;
        }

        /**
         * -----------------------------------------------------------------
         *
         * Routine to extract the <title> from a page HTML.
         *
         * -----------------------------------------------------------------
         */

        function helperExtractTitle ( $html, $default = '' ) {
            $attrs = '(\s[^>]*?)?';
            $start = '~^' .
                         '.*?' .
                         '<html' . $attrs . '>' .
                             '.*?' .
                             '<head' . $attrs . '>' .
                                 '.*?' .
                                 '<title' . $attrs . '>' .
                                     '\s*~uis';
            $end   =                 '~\s*' .
                                 '</title>' .
                                 '.*' .
                             '</head>' .
                             '.*' .
                     '$~uis';
            $test = preg_replace ( $start, '', $html );
            if ( $test != $html ) {
                $html = preg_replace ( $end, '', $test );
                if ( $test != $html ) {
                    $html = preg_replace ( '~\s*</?[^\s>][^>]*?>\s*~u', ' ', $html );
                    $html = preg_replace ( '~\s+~u', ' ', $html );
                    return $html == '' ? $default
                                       : $html;
                }
            }
            return $default;
        }

        /**
         * -----------------------------------------------------------------
         *
         * Routine to extract the first <h1> from a page HTML.
         *
         * -----------------------------------------------------------------
         */

        function helperExtractH1 ( $html, $default = '' ) {
            $attrs = '(\s[^>]*?)?';
            for ( $num = 1; $num <= 6; $num++ ) {
                $start = '~^' .
                             '.*?' .
                             '<h' . $num . $attrs . '>' .
                                 '\s*~uis';
                $end   =         '~\s*' .
                             '</h' . $num . '>' .
                             '.*' .
                         '$~uis';
                $test = preg_replace ( $start, '', $html );
                if ( $test != $html ) {
                    $h1 = preg_replace ( $end, '', $test );
                    if ( $test != $h1 ) {
                        $h1 = preg_replace ( '~\s*</?[^\s>][^>]*?>\s*~u', ' ', $h1 );
                        $h1 = preg_replace ( '~\s+~u', ' ', $h1 );
                        return $h1 == '' ? $default
                                         : $h1;
                    }
                }
            }
            return $default;
        }

        /**
         * -----------------------------------------------------------------
         *
         * Routine to extract the first <p> from a page HTML.
         *
         * -----------------------------------------------------------------
         */

        function helperExtractP ( $html, $default = '' ) {
            $attrs = '(\s[^>]*?)?';
            $start = '~^' .
                         '.*?' .
                         '<p' . $attrs . '>' .
                             '\s*~uis';
            $end   =         '~\s*' .
                         '</p>' .
                         '.*' .
                     '$~uis';
            $test = preg_replace ( $start, '', $html );
            if ( $test != $html ) {
                $html = preg_replace ( $end, '', $test );
                if ( $test != $html ) {
                    $html = preg_replace ( '~\s*</?[^\s>][^>]*?>\s*~u', ' ', $html );
                    $html = preg_replace ( '~\s+~u', ' ', $html );
                    return $html == '' ? $default
                                       : $html;
                }
            }
            return $default;
        }

        /**
         * -----------------------------------------------------------------
         *
         * Routine to extract a <meta> tag from a page HTML.
         *
         * -----------------------------------------------------------------
         */

        function helperExtractMeta ( $html, $name, $default = '' ) {
            $attrs = '(\s[^>]*?)?';
            $start = '~^' .
                         '.*?' .
                         '<html' . $attrs . '>' .
                             '.*?' .
                             '<head' . $attrs . '>' .
                                 '.*?' .
                                 '<meta' . '\s+' . 'name=(["\'])' . preg_quote ( $name, '~' ) . '\3' . '\s*' . 'content="' .
                                 '~uis';
            $end   =             '~"' . $attrs . '>' .
                                 '.*' .
                             '</head>' .
                             '.*' .
                     '$~uis';
            $test = preg_replace ( $start, '', $html );
            if ( $test != $html ) {
                $html = preg_replace ( $end, '', $test );
                if ( $test != $html ) {
                    $html = preg_replace ( '~\s+~u', ' ', $html );
                    return $html == '' ? $default
                                       : $html;
                }
            }
            return $default;
        }

        /**
         * -----------------------------------------------------------------
         *
         * Routine to extract the <link> style URL from a page HTML.
         *
         * -----------------------------------------------------------------
         */

        function helperExtractStyle ( $html, $default = '' ) {
            $attrs1 = '(\s[^>]*?)?';
            $attrs2 = '([\s/][^>]*?)?';
            $start = '~^' .
                         '.*?' .
                         '<html' . $attrs1 . '>' .
                             '.*?' .
                             '<head' . $attrs1 . '>' .
                                 '.*?' .
                                 '<link' . '\s+' . 'rel=(["\'])stylesheet\3' . '\s*' . 'href="' .
                                 '~uis';
            $end   =             '~"' . $attrs2 . '>' .
                                 '.*' .
                             '</head>' .
                             '.*' .
                     '$~uis';
            $test = preg_replace ( $start, '', $html );
            if ( $test != $html ) {
                $html = preg_replace ( $end, '', $test );
                if ( $test != $html ) {
                    return $html == '' ? $default
                                       : $html;
                }
            }
            return $default;
        }

        /**
         * -----------------------------------------------------------------
         *
         * Routine to extract the body from a page HTML.
         *
         * -----------------------------------------------------------------
         */

        function helperExtractBody ( $html, $default = '' ) {
            $attrs = '(\s[^>]*?)?';
            $start = '~^' .
                         '.*?' .
                         '<html' . $attrs . '>' .
                             '.*?' .
                             '<head' . $attrs . '>' .
                                 '.*?' .
                             '</head>' .
                             '\s*' .
                             '<body' . $attrs . '>' .
                                 '\s*' .
                                 '~uis';
            $end   =             '~\s*' .
                             '</body>' .
                             '.*' .
                     '$~uis';
            $test = preg_replace ( $start, '', $html );
            if ( $test != $html ) {
                $html = preg_replace ( $end, '', $test );
                if ( $test != $html ) {
                    return $html == '' ? $default
                                       : $html;
                }
            }
            return $default;
        }

        /**
         * -----------------------------------------------------------------
         *
         * Routine to extract the first <!-- Name: ... --> directive from
         * a page HTML.
         *
         * -----------------------------------------------------------------
         */

        function helperExtractName ( $html, $default = '' ) {
            $tags = [ [ '<!--\s*', '\s*-->'       ] ,
                      [ '/\*+\s*', '\s*\*+/'      ] ,
                      [ '//+\s*',  '[ \t]*[\r\n]' ] ];
            foreach ( $tags as $tag ) {
                $start = '~^' .
                             '.*?' .
                             $tag[ 0 ] . 'name\s*:' .
                                 '\s*~uis';
                $end   =         '~' .
                             $tag[ 1 ] .
                             '.*' .
                         '$~uis';
                $test = preg_replace ( $start, '', $html );
                if ( $test != $html ) {
                    $html = preg_replace ( $end, '', $test );
                    if ( $test != $html ) {
                        $html = preg_replace ( '~\s+~u', ' ', $html );
                        return $html == '' ? $default
                                           : $html;
                    }
                }
            }
            return $default;
        }

        /**
         * -----------------------------------------------------------------
         *
         * Routine to return a page title.
         *
         * -----------------------------------------------------------------
         */

        function helperGetTitle ( $html, $default = '' ) {
            $value = helperExtractTitle ( $html );
            return $value == '' ? helperExtractH1 ( $html, $default )
                                : $value;
        }

        /**
         * -----------------------------------------------------------------
         *
         * Routine to return a page meta description.
         *
         * -----------------------------------------------------------------
         */

        function helperGetMeta ( $html, $default = '' ) {
            $value = helperExtractMeta ( $html, 'description' );
            return $value == '' ? helperExtractP ( $html, $default )
                                : $value;
        }

        /**
         * -----------------------------------------------------------------
         *
         * Routine to return a page permission (a related meta tag).
         *
         * -----------------------------------------------------------------
         */

        function helperGetPermission ( $html, $default = '' ) {
            $value = helperExtractMeta ( $html, 'editing-permission' );
            return $value == '' ? $default
                                : mb_strtolower ( $value, 'UTF-8' );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Routine to return a page H1.
         *
         * -----------------------------------------------------------------
         */

        function helperGetH1 ( $html, $default = '' ) {
            $value = helperExtractH1 ( $html );
            return $value == '' ? helperExtractTitle ( $html, $default )
                                : $value;
        }

        /**
         * -----------------------------------------------------------------
         *
         * Routine to return a file virtual name.
         *
         * -----------------------------------------------------------------
         */

        function helperGetName ( $html, $default = '' ) {
            return helperExtractName ( $html, $default );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Routine to output the page body. It also compiles bbCodes passing
         * each bbCode file all the parameters extracted from its tag.
         *
         * -----------------------------------------------------------------
         */

        function helperEchoBody ( $html ) {
            global $app;
            $body    = helperExtractBody ( $html );
            $pattern = '~^(.*?)'                       .
                           '(?:<(p)(?:\s[^>]*)?>\s*)?' .
                               '\[insert=([^\]\s]*?\.tpl)((?:\s+[^\]\s]+)*)\s*\]' .
                           '(?:\s*</\2>)?' .
                       '(.*)$~uis';
            while ( preg_match ( $pattern, $body ) ) {
                echo           preg_replace ( $pattern, '$1', $body   );
                $url         = preg_replace ( $pattern, '$3', $body   );
                $params      = preg_replace ( $pattern, '$4', $body   );
                $params      = preg_split   ( '~\s+~u',       $params );
                $params[ 0 ] = 'bbcodes/' . $app->database->bbcodes->asFilename ( $url );
                mimimiModule ( count ( $params ) > 1 ? $params
                                                     : $params[ 0 ] );
                $body = preg_replace ( $pattern, '$5', $body );
            }
            echo $body;
        }

        /**
         * -----------------------------------------------------------------
         *
         * Routine to output the page body to edit it. Please note that the
         * RegExp patterns used below are the same as those in the function
         * prepareHtml() declared in the client-side script file
         * "static.pages.only/Themes/default/js/constructor.js". These
         * patterns are used to prevent certain parts of the content from
         * being automatically executed or shown while it is being edited.
         *
         * -----------------------------------------------------------------
         */

        function helperEchoEditableBody ( $entry ) {
            $next = preg_replace ( '~<!--.*?-->~us', '', $entry[ 'body' ] );
            $next = preg_replace ( '~<!--.*$~us',    '', $next            );
            $next = preg_replace ( '~<\?.*?\?>$~us', '', $next            );
            $next = preg_replace ( '~<\?.*$~us',     '', $next            );
            $next = preg_replace ( '~(</?)(safe_)?(!doctype|html|head|body|script|noscript|template|meta|link|iframe|form|math|audio|video|object|embed|applet|frame|noframes)([\s=/>])~ui', '$1safe_$2$3$4', $next );
            $next = preg_replace ( '~</([a-z][^\s=/>]*)[^>]*?>~ui', '</$1>', $next );
            do {
                $body = $next;
                $next = preg_replace ( '~(<[a-z][^>]*?[\s="\'/])((on[a-z0-9_]*|href)[\s=/>])~ui', '$1safe_$2', $body );
            } while ( $next != $body );
            echo $body;
        }

        /**
         * -----------------------------------------------------------------
         *
         * Routine to print a URL of the page style file.
         *
         * -----------------------------------------------------------------
         */

        function helperPrintStyleUrl ( $html, $default = 'default.css' ) {
            $value = helperExtractStyle ( $html, $default );
            printThemeUrl (                       );
            printValue    ( 'css/pages/' . $value );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Routine to print a page title.
         *
         * -----------------------------------------------------------------
         */

        function helperPrintTitle ( $html, $default = '' ) {
            $value = helperGetTitle ( $html, $default );
            printValue ( $value );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Routine to print a page meta description.
         *
         * -----------------------------------------------------------------
         */

        function helperPrintMeta ( $html, $default = '' ) {
            $value = helperGetMeta ( $html, $default );
            printValue ( $value );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Routine to print a page title.
         *
         * -----------------------------------------------------------------
         */

        function helperPrintH1 ( $html, $default = '' ) {
            $value = helperGetH1 ( $html, $default );
            printValue ( $value );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Routine to return the static page URL (non-administrative).
         *
         * -----------------------------------------------------------------
         */

        function helperGetPageUrl ( $url = NULL ) {
            $url = is_string ( $url ) ? $url
                                      : mimimiUri ( FALSE );
            return preg_replace ( '~^(pages|add|construct|copy|delete|edit|markup|upload|download)(/|$)+~ui', '', $url );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Routine to return the CSS file URL (non-administrative).
         *
         * -----------------------------------------------------------------
         */

        function helperGetCssUrl ( $url = NULL ) {
            $url = is_string ( $url ) ? $url
                                      : mimimiUri ( FALSE );
            return preg_replace ( '~^styles/(add|copy|delete|edit|upload|download)(/|$)+~ui', '', $url );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Routine to return the bbCode file URL (non-administrative).
         *
         * -----------------------------------------------------------------
         */

        function helperGetBBcodeUrl ( $url = NULL ) {
            $url = is_string ( $url ) ? $url
                                      : mimimiUri ( FALSE );
            return preg_replace ( '~^bbcodes/(add|copy|delete|edit|upload|download)(/|$)+~ui', '', $url );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Routine to return the Fragment file URL (non-administrative).
         *
         * -----------------------------------------------------------------
         */

        function helperGetFragmentUrl ( $url = NULL ) {
            $url = is_string ( $url ) ? $url
                                      : mimimiUri ( FALSE );
            return preg_replace ( '~^fragments/(add|copy|delete|edit|upload|download)(/|$)+~ui', '', $url );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Routine to check if an administrative URL contains a ADD command.
         *
         * -----------------------------------------------------------------
         */

        function helperIsAddUrl ( $url = NULL ) {
            $url = is_string ( $url ) ? $url
                                      : mimimiUri ( FALSE );
            return preg_match ( '~^((bbcodes|fragments|pages|styles)/)?add(/|$)+~ui', $url );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Routine to check if an administrative URL contains a CONSTRUCT command.
         *
         * -----------------------------------------------------------------
         */

        function helperIsConstructUrl ( $url = NULL ) {
            $url = is_string ( $url ) ? $url
                                      : mimimiUri ( FALSE );
            return preg_match ( '~^((bbcodes|fragments|pages|styles)/)?construct(/|$)+~ui', $url );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Routine to check if an administrative URL contains a COPY command.
         *
         * -----------------------------------------------------------------
         */

        function helperIsCopyUrl ( $url = NULL ) {
            $url = is_string ( $url ) ? $url
                                      : mimimiUri ( FALSE );
            return preg_match ( '~^((bbcodes|fragments|pages|styles)/)?copy(/|$)+~ui', $url );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Routine to check if an administrative URL contains a EDIT command.
         *
         * -----------------------------------------------------------------
         */

        function helperIsEditUrl ( $url = NULL ) {
            $url = is_string ( $url ) ? $url
                                      : mimimiUri ( FALSE );
            return preg_match ( '~^((bbcodes|fragments|pages|styles)/)?edit(/|$)+~ui', $url );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Routine to check if an administrative URL contains a MARKUP command.
         *
         * -----------------------------------------------------------------
         */

        function helperIsMarkupUrl ( $url = NULL ) {
            $url = is_string ( $url ) ? $url
                                      : mimimiUri ( FALSE );
            return preg_match ( '~^((bbcodes|fragments|pages|styles)/)?markup(/|$)+~ui', $url );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Routine to check if an administrative URL contains a UPLOAD command.
         *
         * -----------------------------------------------------------------
         */

        function helperIsUploadUrl ( $url = NULL ) {
            $url = is_string ( $url ) ? $url
                                      : mimimiUri ( FALSE );
            return preg_match ( '~^((bbcodes|fragments|pages|styles)/)?upload(/|$)+~ui', $url );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Routine to check if an administrative URL contains a DOWNLOAD command.
         *
         * -----------------------------------------------------------------
         */

        function helperIsDownloadUrl ( $url = NULL ) {
            $url = is_string ( $url ) ? $url
                                      : mimimiUri ( FALSE );
            return preg_match ( '~^((bbcodes|fragments|pages|styles)/)?download(/|$)+~ui', $url );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Routine to retrieve value of the form field as a boolean.
         *
         * -----------------------------------------------------------------
         */

        function helperGetFormFlag ( $inputName ) {
            $value = getFormInput ( $inputName );
            return ! empty ( $value );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Routine to retrieve value of the form field as a session mode.
         *
         * -----------------------------------------------------------------
         */

        function helperGetFormMode ( $inputName ) {
            $value = getFormInput ( $inputName );
            return $value === '' ? ''
                                 : ! empty ( $value );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Routine to retrieve value of the form field as a number.
         *
         * -----------------------------------------------------------------
         */

        function helperGetFormNumber ( $inputName, $min = 0, $max = 300 ) {
            $value = getFormInput ( $inputName, 0 );
            $value = @ intval ( $value );
            return min ( $max, max ( $min, $value ) );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Routine to retrieve value of the form field as a multiline string.
         *
         * -----------------------------------------------------------------
         */

        function helperGetFormText ( $inputName ) {
            $value = mimimiPost ( $inputName );
            $value = preg_replace ( '~(^[\r\n]+|\s+$)~u',                     '',     $value );
            $value = preg_replace ( '~[ \t]+([\r\n])~u',                      '$1',   $value );
            $value = preg_replace ( preg_match ( '~\ru~', $value ) ? '~\n~u'
                                                                   : '~\r~u', '',     $value );
            $value = preg_replace ( '~(\r\r|\n\n)[\r\n]+~u',                  '$1',   $value );
            return   preg_replace ( '~[\r\n]~u',                              "\r\n", $value );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Routine to retrieve value of the form field as a multiline JavaScript.
         *
         * -----------------------------------------------------------------
         */

        function helperGetFormJs ( $inputName ) {
            $value = helperGetFormText ( $inputName );
            return   helperRemoveSSI ( $value );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Routine to retrieve value of the form field as a multiline CSS.
         *
         * -----------------------------------------------------------------
         */

        function helperGetFormCss ( $inputName ) {
            $value = helperGetFormText ( $inputName );
            return   helperRemoveSSI ( $value );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Routine to retrieve value of the form field as a multiline HTML template.
         *
         * -----------------------------------------------------------------
         */

        function helperGetFormTpl ( $inputName, $noPhp = TRUE ) {
            $value = helperGetFormText ( $inputName );
            return   helperRemoveSSI ( $value, $noPhp );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Routine to redirect the current visitor to a page.
         *
         * -----------------------------------------------------------------
         */

        function helperGotoPage ( $path = '' ) {
            $home = printSiteUrl ( FALSE );
            mimimiStop ( $home . $path, 307 );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Routine to redirect the current visitor to the home page.
         *
         * -----------------------------------------------------------------
         */

        function helperGotoHome ( ) {
            helperGotoPage ( );
        }
