<?php
/**
 * -------------------------------------------------------------------------
 *
 * @package    MimimiFramework
 * @subpackage Examples / Shopping UI with Vue.js
 * @license    GPL-2.0
 *             https://opensource.org/license/gpl-2-0/
 * @copyright  2022 MiMiMi Community
 *             https://mimimi.software/
 *
 * -------------------------------------------------------------------------
 */

    mimimiInclude ( 'Helper/Helper.php' );
    class MyMimimiHelper extends MimimiHelper {

        /**
         * -----------------------------------------------------------------
         *
         * Returns the contents of a Vue template.
         *
         * -----------------------------------------------------------------
         *
         * This method extracts the contents of specified template using a
         * system routine "mimimiDump". That template resolves relative to
         * "Themes/default" directory.
         *
         * This method then uses the same system routine to append the
         * contents of the site's main stylesheet located at "css/styles.css".
         * Also relative to "Themes/default" directory.
         *
         * -----------------------------------------------------------------
         *
         * If the result is to be used as a Javascript literal delimited
         * with backtick characters (that is Template Literal), the dollar
         * signs and backticks in that result must be escaped with a
         * backslash.
         *
         * If the result is to be used as a Javascript literal enclosed in
         * double quotes (that is String Literal), the dollar signs and
         * quotes in that result must be escaped with a backslash.
         *
         * -----------------------------------------------------------------
         *
         * @public
         * @param   string  $file  Name of the required Vue file.
         * @return  string
         *
         * -----------------------------------------------------------------
         */

        public function readVue ( $file ) {
            return mimimiDump ( $file, FALSE, FALSE )                .
                   '<style>'                                         .
                       mimimiDump ( 'css/styles.css', FALSE, FALSE ) .
                   '</style>';
        }

        /**
         * -----------------------------------------------------------------
         *
         * Escapes text for use as a Javascript string literal.
         *
         * -----------------------------------------------------------------
         *
         * @public
         * @param   string  $text  The text to be escaped.
         * @return  string
         *
         * -----------------------------------------------------------------
         */

        public function asStringLiteral ( $text ) {
            $js  = $this->asLiteral ( $text );
            return $this->splitTags ( $js   );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Escapes text for use as a Javascript template literal.
         *
         * -----------------------------------------------------------------
         *
         * @public
         * @param   string  $text  The text to be escaped.
         * @return  string
         *
         * -----------------------------------------------------------------
         */

        public function asTemplateLiteral ( $text ) {
            $js  = $this->asLiteral ( $text, '~([`$\\\\])~u' );
            return $this->splitTags ( $js                    );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Escapes text for use as a string literal.
         *
         * -----------------------------------------------------------------
         *
         * @protected
         * @param   string  $text  The text to be escaped.
         * @return  string
         *
         * -----------------------------------------------------------------
         */

        protected function asLiteral ( $text, $pattern = '~(["$\r\n\\\\])~u' ) {
            $js  = preg_replace     ( $pattern, '\\\$1', $text );
            return $this->splitTags ( $js                      );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Splits HTML tags in a Javascript string.
         *
         * -----------------------------------------------------------------
         *
         * @protected
         * @param   string  $js  The string to be split.
         * @return  string
         *
         * -----------------------------------------------------------------
         */

        protected function splitTags ( $js ) {
            $js  = preg_replace ( '~<([!?/a-z])~ui',    "<\\\r\n\$1", $js );
            return preg_replace ( '~([/?a-z\'"-])>~ui', "\$1\\\r\n>", $js );
        }
    };
