/**
 * -------------------------------------------------------------------------
 *
 * Script for working with a menu or toolbar on the client side.
 *
 * -------------------------------------------------------------------------
 *
 * @package    MimimiFramework
 * @subpackage Examples / IDE skeleton
 * @license    GPL-2.0
 *             https://opensource.org/license/gpl-2-0/
 * @copyright  2022 MiMiMi Community
 *             https://mimimi.software/
 *
 * -------------------------------------------------------------------------
 */

    class AppMenubar {

        /**
         * -----------------------------------------------------------------
         *
         * Public properties.
         *
         * -----------------------------------------------------------------
         */

        links = null;

        /**
         * -----------------------------------------------------------------
         *
         * Private properties.
         *
         * -----------------------------------------------------------------
         */

        #root = null;

        /**
         * -----------------------------------------------------------------
         *
         * Creates an instance of the class.
         *
         * -----------------------------------------------------------------
         */

        constructor ( selector ) {
            this.#root = document.querySelector ( selector );
            if ( this.#root ) {
                this.links = this.#root.querySelectorAll ( '.link' );
                this.links.forEach (
                    ( node ) => {
                        node.onclick     = this.onClick;
                        node.onmousedown = this.onHover;
                        node.onmousemove = this.onHover;
                        node.onfocus     = this.onHover;
                    }
                );
                document.body.addEventListener ( 'mousemove', this.onLeave );
                document.body.addEventListener ( 'keydown',   this.onKey   );
                document.body.addEventListener ( 'keyup',     this.onKey   );
            }
        }

        /**
         * -----------------------------------------------------------------
         *
         * Retrieves the node of a menu or toolbar link by its path and shortcut.
         *
         * -----------------------------------------------------------------
         */

        lookForLink ( path, shortcut = '' ) {
            const opening  = '.item[data-name="',
                  ending   = '"]',
                  link     = shortcut != ''
                                      ?  '.link[data-key="' +
                                             shortcut +
                                         ending
                                      :  '> .link',
                  selector = opening +
                                 path.replace (
                                     /\s*\/+\s*/g,
                                     ending + ' ' + opening
                                 ) +
                             ending + ' ' + link;
            return this.#root.querySelector ( selector );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Retrieves the node of a menu or toolbar link by its event.
         *
         * -----------------------------------------------------------------
         */

        findLink ( event ) {
            event.preventDefault ( );
            event.stopPropagation ( );
            return event.target.closest ( 'button' );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Handles the keyboard shortcuts in the menu or toolbar.
         *
         * -----------------------------------------------------------------
         */

        onKey ( event ) {
            if ( event.altKey ) {
                if ( ! event.shiftKey ) {
                    if ( ! event.ctrlKey ) {
                        let plan     = [ 'topmenu', 'toolbar' ],
                            j        = 0,
                            key      = event.key.toLowerCase ( ),
                            state    = '',
                            shortcut = '';
                        while ( j < plan.length ) {
                            let i = 0;
                            while ( i < app[ plan[ j ] ].links.length ) {
                                shortcut = app[ plan[ j ] ].links[ i ].getAttribute ( 'data-key' );
                                if ( typeof shortcut == 'string' ) {
                                    if ( shortcut.toLowerCase ( ) == key ) {
                                        event.preventDefault ( );
                                        event.stopPropagation ( );
                                        state = app[ plan[ j ] ].links[ i ].getAttribute ( 'disabled' );
                                        if ( state == null ) {
                                            if ( event.type == 'keyup' ) {
                                                app[ plan[ j ] ].links[ i ].click ( );
                                            }
                                            return;
                                        }
                                    }
                                }
                                i++;
                            }
                            j++;
                        }
                    }
                }
            }
        }

        /**
         * -----------------------------------------------------------------
         *
         * Handles focus outside of a menu or toolbar link.
         *
         * -----------------------------------------------------------------
         */

        onLeave ( event ) {
            event.preventDefault ( );
            event.stopPropagation ( );
            app.statusbar.showMessage ( );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Handles focus on a menu or toolbar link.
         *
         * -----------------------------------------------------------------
         */

        onHover ( event ) {
            const node = app.topmenu.findLink ( event );
            if ( node ) {
                const title = node.getAttribute ( 'data-title' );
                              app.statusbar.showMessage ( title === null ? '' : title );
            }
        }

        /**
         * -----------------------------------------------------------------
         *
         * Handles a click on a menu or toolbar link.
         *
         * -----------------------------------------------------------------
         */

        onClick ( event ) {
            const node = app.topmenu.findLink ( event );
            if ( node ) {
                app.removeModal ( );
                const url = node.getAttribute ( 'data-href' );
                      if ( url !== null ) {
                          app.statusbar.showMessage ( '' );
                          node.blur ( );
                          if ( ! url.match ( /^\s*javascript:\s*/ ) ) {
                              document.body.classList.add ( 'loading' );
                          }
                          window.location.replace ( url );
                      }
            }
        };

        /**
         * -----------------------------------------------------------------
         *
         * Enables/disables a link specified by its path and shortcut.
         *
         * -----------------------------------------------------------------
         */

        switchLink ( path, shortcut = '', enable = true ) {
            const node = this.lookForLink ( path, shortcut );
            if ( node ) {
                const attr = 'disabled';
                enable ? node.removeAttribute ( attr       )
                       : node.setAttribute    ( attr, attr );
            }
        }

        /**
         * -----------------------------------------------------------------
         *
         * Replaces a link URL specified by its path and shortcut.
         *
         * -----------------------------------------------------------------
         */

        replaceUrl ( url, path, shortcut = '' ) {
            const node = this.lookForLink ( path, shortcut );
            if ( node ) {
                node.setAttribute ( 'data-href', url );
            }
        }
    };

    /**
     * ---------------------------------------------------------------------
     *
     * Initializes the topmenu.
     *
     * ---------------------------------------------------------------------
     */

    if ( typeof app != 'undefined' ) {
        app.topmenu = new AppMenubar ( '.screen > .menu'    );
        app.toolbar = new AppMenubar ( '.screen > .toolbar' );
    }
