/**
 * -------------------------------------------------------------------------
 *
 * Script for working with the status bar on the client side.
 *
 * -------------------------------------------------------------------------
 *
 * @package    MimimiFramework
 * @subpackage Examples / IDE skeleton
 * @license    GPL-2.0
 *             https://opensource.org/license/gpl-2-0/
 * @copyright  2022 MiMiMi Community
 *             https://mimimi.software/
 *
 * -------------------------------------------------------------------------
 */

    class AppStatusbar {

        /**
         * -----------------------------------------------------------------
         *
         * Private properties.
         *
         * -----------------------------------------------------------------
         */

        #timer    = null;
        #root     = null;
        #elements = {
            msg:    null,
            mode:   null,
            cursor: null,
            size:   null,
            clock:  null
        };

        /**
         * -----------------------------------------------------------------
         *
         * Creates an instance of the class.
         *
         * -----------------------------------------------------------------
         */

        constructor ( selector ) {
            this.#root = document.querySelector ( selector );
            if ( this.#root ) {
                for ( const name in this.#elements ) {
                    let node = this.#root.querySelector ( '.' + name );
                        node.onmousemove = this.onHover;
                    this.#elements[ name ] = node;
                }
            }
        }

        /**
         * -----------------------------------------------------------------
         *
         * Handles focus on a statusbar element.
         *
         * -----------------------------------------------------------------
         */

        onHover ( event ) {
            event.preventDefault ( );
            event.stopPropagation ( );
            const title = this.getAttribute ( 'data-title' );
                          app.statusbar.showMessage ( title === null ? '' : title );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Returns the current time in HH:MM or HH:MM:SS format.
         *
         * -----------------------------------------------------------------
         */

        getTime ( withSeconds = false ) {
            const  clock = new Date ( ),
                   size  = withSeconds ? 8 : 5;
            return clock.toTimeString ( ).substr ( 0, size );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Displays the current time in HH:MM or HH:MM:SS format.
         *
         * -----------------------------------------------------------------
         */

        showTime ( withSeconds = false ) {
            const value = app.statusbar.getTime ( withSeconds );
                          app.statusbar.showClock ( value );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Stops displaying the clock if running.
         *
         * -----------------------------------------------------------------
         */

        stopClock ( ) {
            if ( this.#timer ) {
                clearInterval ( this.#timer );
                this.#timer = null;
            }
            this.showClock ( );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Starts displaying the clock.
         *
         * -----------------------------------------------------------------
         */

        startClock ( period = 1000 ) {
            this.stopClock ( );
            this.showTime  ( );
            this.#timer = setInterval ( this.showTime, period );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Displays an element.
         *
         * -----------------------------------------------------------------
         */

        showMessage ( value = '' ) { if ( this.#elements.msg    ) this.#elements.msg.innerText    = value }
        showMode    ( value = '' ) { if ( this.#elements.mode   ) this.#elements.mode.innerText   = value }
        showCursor  ( value = '' ) { if ( this.#elements.cursor ) this.#elements.cursor.innerText = value }
        showSize    ( value = '' ) { if ( this.#elements.size   ) this.#elements.size.innerText   = value }
        showClock   ( value = '' ) { if ( this.#elements.clock  ) this.#elements.clock.innerText  = value }
    };

    /**
     * ---------------------------------------------------------------------
     *
     * Initializes the status bar.
     *
     * ---------------------------------------------------------------------
     */

    if ( typeof app != 'undefined' ) {
        app.statusbar = new AppStatusbar ( '.screen > .statusbar' );
        app.statusbar.startClock ( );
    }
