<?php
/**
 * -------------------------------------------------------------------------
 *
 * The module for verifying administrative access, which was implemented
 * using a constants file.
 *
 * -------------------------------------------------------------------------
 *
 * It implements the following methods:
 *
 *     PUBLIC  run            ( [$prefix]      )  -->  to provide a constants prefix used by your application
 *     PUBLIC  runSession     (                )  -->  to launch the basic SESSION module
 *     PUBLIC  getFromSession ( $param[, $def] )  -->  to get the value stored in session
 *     PUBLIC  putToSession   ( $param, $value )  -->  to store the value in session
 *     PUBLIC  getMyLabel     (                )  -->  to get a label identifying you (the current visitor)
 *     PUBLIC  getAdminLabel  (                )  -->  to get a label identifying the last logged in admin
 *     PUBLIC  setAdminLabel  ( $label         )  -->  to store a label identifying the last logged in admin
 *     PUBLIC  getAdminId     (                )  -->  to get an ID identifying the last logged in admin
 *     PUBLIC  setAdminId     ( $id            )  -->  to store an ID identifying the last logged in admin
 *     PUBLIC  getTimeout     ( $type          )  -->  to get the last timeout period specified by type
 *     PUBLIC  setTimeout     ( $type, $pause  )  -->  to store a new timeout period (timer pause) specified by type
 *     PUBLIC  computeHash    ( $password      )  -->  to return the MD5 hash of a password
 *     PUBLIC  amIAdmin       (                )  -->  to check if you (the current visitor) are an administrator
 *     PUBLIC  getConstant    ( $name[, $def]  )  -->  to return the value of a prefixed constant
 *     PUBLIC  logIn          ( $id            )  -->  to log in an admin
 *     PUBLIC  logOut         (                )  -->  to log out the admin immediately
 *
 * Note that if you want to use this module in your application, you will
 * need to create the following constants in it:
 *
 *     MYAPP_SESSION_MODE
 *     MYAPP_DEMO_MODE
 *
 * Please note also that MYAPP_ prefix is used by default. However, if you
 * plan to use your app in orchestration workflows, you should replace this
 * prefix with a unique one to avoid conflicts when two applications have
 * constants with the same name.
 *
 * -------------------------------------------------------------------------
 *
 * @package    MimimiFramework
 * @subpackage Modules
 * @license    GPL-2.0
 *             https://opensource.org/license/gpl-2-0/
 * @copyright  2022 MiMiMi Community
 *             https://mimimi.software/
 *
 * -------------------------------------------------------------------------
 */

    mimimiInclude ( 'Module.php' );

    class MimimiSecurity extends MimimiModule {

        /**
         * -----------------------------------------------------------------
         *
         * Specify a constants prefix used by default.
         *
         * -----------------------------------------------------------------
         *
         * @access protected
         * @var    string
         *
         * -----------------------------------------------------------------
         */

        protected $constantPrefix = 'MYAPP_';

        /**
         * -----------------------------------------------------------------
         *
         * Allows you to specify a constants prefix used by your app.
         *
         * -----------------------------------------------------------------
         *
         * Please note that this method should be called once and only if
         * your application uses a non-default constants prefix (the default
         * is MYAPP_).
         *
         * -----------------------------------------------------------------
         *
         * @public
         * @param   string  $prefix  (optional) The prefix to be used.
         * @return  void
         *
         * -----------------------------------------------------------------
         */

        public function run ( $prefix = '' ) {
            $this->constantPrefix = $prefix;
        }

        /**
         * -----------------------------------------------------------------
         *
         * Launches the SESSION module.
         *
         * -----------------------------------------------------------------
         *
         * @public
         * @return  void
         *
         * -----------------------------------------------------------------
         */

        public function runSession ( ) {
            $mode = $this->getConstant ( 'SESSION_MODE' );
            $this->app->session->run ( $mode );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Retrieves the value stored in a session parameter.
         *
         * -----------------------------------------------------------------
         *
         * @public
         * @param   string  $param    The parameter name.
         * @param   mixed   $default  (optional) Default value if there is no such parameter.
         * @return  mixed
         *
         * -----------------------------------------------------------------
         */

        public function getFromSession ( $param, $default = NULL ) {
            $this->runSession ( );
            return $this->app->session->get ( $param, $default );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Stores the value in a session parameter.
         *
         * -----------------------------------------------------------------
         *
         * @public
         * @param   string  $param  The parameter name.
         * @param   mixed   $value  The value to be saved.
         * @return  void
         *
         * -----------------------------------------------------------------
         */

        public function putToSession ( $param, $value ) {
            $this->runSession ( );
            return $this->app->session->set ( $param, $value );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Retrieves a label identifying the current visitor.
         *
         * -----------------------------------------------------------------
         *
         * Note that we compute this label as the MD5-hashed IP to prevent
         * session hijacking by spoofing its ID (for more details, see the
         * amIAdmin() method below).
         *
         * -----------------------------------------------------------------
         *
         * @public
         * @return  string
         *
         * -----------------------------------------------------------------
         */

        public function getMyLabel ( ) {
            $ip = mimimiServer ( 'REMOTE_ADDR' );
            return md5 ( $ip );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Retrieves a label identifying the last admin.
         *
         * -----------------------------------------------------------------
         *
         * @public
         * @return  string|null  A string label if that admin is still online.
         *                       EMPTY STRING   if that admin is already logged out.
         *                       NULL           if there is no last admin.
         *
         * -----------------------------------------------------------------
         */

        public function getAdminLabel ( ) {
            return $this->getFromSession ( 'adminLabel' );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Stores a label identifying the last admin.
         *
         * -----------------------------------------------------------------
         *
         * @public
         * @param   string  $label  A label to be saved.
         * @return  void
         *
         * -----------------------------------------------------------------
         */

        public function setAdminLabel ( $label ) {
            $this->putToSession ( 'adminLabel', $label );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Retrieves an ID identifying the last admin.
         *
         * -----------------------------------------------------------------
         *
         * @public
         * @return  int|null  An identifier if that admin is still online.
         *                    ZERO          if that admin is already logged out.
         *                    NULL          if there is no last admin.
         *
         * -----------------------------------------------------------------
         */

        public function getAdminId ( ) {
            return $this->getFromSession ( 'adminId' );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Stores an ID identifying the last admin.
         *
         * -----------------------------------------------------------------
         *
         * @public
         * @param   int  $id  An identifier to be saved.
         * @return  void
         *
         * -----------------------------------------------------------------
         */

        public function setAdminId ( $id ) {
            $this->putToSession ( 'adminId', $id );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Retrieves the last timeout period.
         *
         * -----------------------------------------------------------------
         *
         * @public
         * @param   string  $type  Name of the timeout type.
         * @return  int            Time stamp for the expiration of this period.
         *
         * -----------------------------------------------------------------
         */

        public function getTimeout ( $type ) {
            return $this->getFromSession ( 'activityTimeout_' . $type, time ( ) );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Stores a new timeout period.
         *
         * -----------------------------------------------------------------
         *
         * @public
         * @param   string  $type   Name of the timeout type.
         * @param   int     $pause  A timer in seconds.
         * @return  void
         *
         * -----------------------------------------------------------------
         */

        public function setTimeout ( $type, $pause ) {
            $this->putToSession ( 'activityTimeout_' . $type, time ( ) + $pause );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Returns the MD5 hash of a password.
         *
         * -----------------------------------------------------------------
         *
         * @public
         * @param   string  $pass  A password to be hashed.
         * @return  string
         *
         * -----------------------------------------------------------------
         */

        public function computeHash ( $pass ) {
            return md5 ( $pass );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Checks if the current visitor is an administrator.
         *
         * -----------------------------------------------------------------
         *
         * Note that in demo mode, any visitor is considered Admin if he
         * visits your site for the first time. We detect this case by
         * matching the admin label against a NULL value.
         *
         * -----------------------------------------------------------------
         *
         * @public
         * @return  bool
         *
         * -----------------------------------------------------------------
         */

        public function amIAdmin ( ) {
            $my    = $this->getMyLabel    ( );
            $label = $this->getAdminLabel ( );
            if ( $label === NULL ) {
                $label = '';
                $id    = 0;
                $demo  = $this->getConstant ( 'DEMO_MODE', TRUE );
                if ( $demo ) {
                    $label = $my;
                    $id    = 1;
                }
                $this->setAdminLabel ( $label );
                $this->setAdminId    ( $id    );
            }
            return $label === $my;
        }

        /**
         * -----------------------------------------------------------------
         *
         * Returns the value of a prefixed constant.
         *
         * -----------------------------------------------------------------
         *
         * @public
         * @param   string  $name     The constant name.
         * @param   mixed   $default  (optional) Default value if there is no such constant.
         * @return  mixed
         *
         * -----------------------------------------------------------------
         */

        public function getConstant ( $name, $default = '' ) {
            $name = $this->constantPrefix . $name;
            return defined ( $name ) ? constant ( $name )
                                     : $default;
        }

        /**
         * -----------------------------------------------------------------
         *
         * Logs in an administrator with the provided identifier.
         *
         * -----------------------------------------------------------------
         *
         * @public
         * @param   int  $id  The admin's identifier.
         * @return  void
         *
         * -----------------------------------------------------------------
         */

        public function logIn ( $id ) {
            $my = $this->getMyLabel (     );
            $this->setAdminLabel    ( $my );
            $this->setAdminId       ( $id );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Logs out the current administrator immediately.
         *
         * -----------------------------------------------------------------
         *
         * @public
         * @return  void
         *
         * -----------------------------------------------------------------
         */

        public function logOut ( ) {
            $this->setAdminLabel ( '' );
            $this->setAdminId    ( 0  );
        }
    };
