<?php
/**
 * -------------------------------------------------------------------------
 * The URL module to work with page address.
 *
 * -------------------------------------------------------------------------
 *
 * Implemented methods below are:
 *     cutSegment
 *     getSegment
 *     pasteSegment
 *     cutParameter
 *     getParameter
 *     pasteParameter
 *     cutPaginator
 *     getPaginator
 *     pastePaginator
 *
 * -------------------------------------------------------------------------
 *
 * @package    MimimiFramework
 * @subpackage Modules
 * @license    GPL-2.0
 *             https://opensource.org/license/gpl-2-0/
 * @copyright  2022 MiMiMi Community
 *             https://mimimi.software/
 * -------------------------------------------------------------------------
 */

mimimiInclude('Module.php');
class MimimiUrl extends MimimiModule {

    /**
     * ---------------------------------------------------------------------
     * Public method to trim the first segment of the URL.
     *
     * @public
     * @param   string  $url    Page address (will be without the first segment on exit).
     * @param   bool    $lcase  TRUE if you want the result to be in lower case.
     * @return  string          Cropped segment.
     * ---------------------------------------------------------------------
     */

    public function cutSegment ( & $url, $lcase = TRUE ) {
        $pattern = '~^([^/\\\\]*)[/\\\\]+(.*?)$~us';
        $segment = preg_replace($pattern, '$1', $url);
        $url     = $segment != $url
                   ? preg_replace($pattern, '$2', $url)
                   : '';
        return $lcase
               ? mb_strtolower($segment, 'UTF-8')
               : $segment;
    }

    /**
     * ---------------------------------------------------------------------
     * Public method to retrieve the first segment of the URL.
     *
     * @public
     * @param   string  $url    Page address.
     * @param   bool    $lcase  TRUE if you want the result to be in lower case.
     * @return  string          Retrieved segment.
     * ---------------------------------------------------------------------
     */

    public function getSegment ( $url, $lcase = TRUE ) {
        return $this->cutSegment($url, $lcase);
    }

    /**
     * ---------------------------------------------------------------------
     * Public method to paste the first segment in the URL.
     *
     * @public
     * @param   string  $url      Page address.
     * @param   string  $segment  Segment to paste.
     * @return  string            Same page address with the first segment.
     * ---------------------------------------------------------------------
     */

    public function pasteSegment ( $url, $segment ) {
        $pattern = '~(^[/\\\\]+|[/\\\\]+$)~u';
        $url     = preg_replace($pattern, '', $url);
        $segment = preg_replace($pattern, '', $segment);
        return preg_replace($pattern, '', $segment . '/'. $url);
    }

    /**
     * ---------------------------------------------------------------------
     * Public method to trim the last segment of the URL.
     *
     * @public
     * @param   string  $url    Page address (will be without the last segment on exit).
     * @param   bool    $lcase  TRUE if you want the result to be in lower case.
     * @return  string          Cropped segment.
     * ---------------------------------------------------------------------
     */

    public function cutParameter ( & $url, $lcase = TRUE ) {
        $pattern = '~^(.*?)[/\\\\]+([^/\\\\]*)$~us';
        $segment = preg_replace($pattern, '$2', $url);
        $url     = $segment != $url
                   ? preg_replace($pattern, '$1', $url)
                   : '';
        return $lcase
               ? mb_strtolower($segment, 'UTF-8')
               : $segment;
    }

    /**
     * ---------------------------------------------------------------------
     * Public method to retrieve the last segment of the URL.
     *
     * @public
     * @param   string  $url    Page address.
     * @param   bool    $lcase  TRUE if you want the result to be in lower case.
     * @return  string          Retrieved segment (in lowercase).
     * ---------------------------------------------------------------------
     */

    public function getParameter ( $url, $lcase = TRUE ) {
        return $this->cutParameter($url, $lcase);
    }

    /**
     * ---------------------------------------------------------------------
     * Public method to paste the last segment in the URL.
     *
     * @public
     * @param   string  $url    Page address.
     * @param   string  $param  Segment to paste.
     * @return  string          Same page address with the last segment.
     * ---------------------------------------------------------------------
     */

    public function pasteParameter ( $url, $param ) {
        $pattern = '~(^[/\\\\]+|[/\\\\]+$)~u';
        $url   = preg_replace($pattern, '', $url);
        $param = preg_replace($pattern, '', $param);
        return preg_replace($pattern, '', $url . '/'. $param);
    }

    /**
     * ---------------------------------------------------------------------
     * Public method to trim the paginator at the end of the URL.
     *
     * @public
     * @param   string   $url     Page address (will be without a paginator on exit).
     * @param   string   $prefix  Several characters before the paginator (default is "page-").
     * @return  int               Page number (from 1 to maximum).
     * ---------------------------------------------------------------------
     */

    public function cutPaginator ( & $url, $prefix = 'page-' ) {
        $pattern = '~^(.*?)[/\\\\]*' . preg_quote($prefix, '~') . '(\d+)$~uis';
        $test = preg_replace($pattern, '$2', $url);
        if ( $test != $url ) {
            $url = preg_replace($pattern, '$1', $url);
            return max(1, intval($test));
        }
        return 1;
    }

    /**
     * ---------------------------------------------------------------------
     * Public method to retrieve the paginator at the end of the URL.
     *
     * @public
     * @param   string   $url     Page address.
     * @param   string   $prefix  Several characters before the paginator (default is "page-").
     * @return  int               Retrieved page number (from 1 to maximum).
     * ---------------------------------------------------------------------
     */

    public function getPaginator ( $url, $prefix = 'page-' ) {
        return $this->cutPaginator($url, $prefix);
    }

    /**
     * ---------------------------------------------------------------------
     * Public method to paste the paginator at the end of the URL.
     *
     * @public
     * @param   string   $url      Page address.
     * @param   int      $pagenum  Paginator to paste.
     * @param   string   $prefix   Several characters before the paginator (default is "page-").
     * @return  string             Same page address with the paginator.
     * ---------------------------------------------------------------------
     */

    public function pastePaginator ( $url, $pagenum, $prefix = 'page-' ) {
        return $this->pasteParameter($url, $prefix . $pagenum);
    }
};