<?php
/**
 * -------------------------------------------------------------------------
 *
 * Главный модуль приложения.
 *
 * -------------------------------------------------------------------------
 *
 * Фреймворк устроен так, что любые запросы к серверу, которые не указывают
 * на реально существующий статический файл (наподобие картинки, стиля,
 * явапскрипта и тому подобных), всегда заворачиваются на главный модуль
 * приложения, а именно его общедоступный метод run().
 *
 * В классе главного модуля, реализованного ниже, имеются следующие методы
 * для обработки входящих запросов:
 *
 *     PUBLIC     run      ( [$params] )  -->  чтобы сгенерировать страницу, запрошенную браузером
 *     PROTECTED  routeFor ( $url      )  -->  чтобы маршрутизировать запрос к его исполнителю согласно предоставленного URL-а
 *
 * -------------------------------------------------------------------------
 *
 * Чтобы вы представляли себе путь движения запроса по файлам и модулям
 * сайта, нарисую его схему. Здесь стрелки вправо обозначают порядок
 * загрузки файлов сайта и выполнения их методов, а стрелки влево - какие
 * сущности декларировались загруженным файлом для использования в вашем
 * приложении.
 *
 *     ├─> .htaccess
 *     │
 *     │   ┌─<─ несколько констант вида MIMIMI_...
 *     │   ├─<─ процедура mimimiLoad()
 *     │   │
 *     └─> index.php
 *           │
 *           ├─> mimimi.core/Config.php
 *           │               │
 *           │               └─<─ ещё несколько констант вида MIMIMI_...
 *           │
 *           ├─> mimimi.core/RoutinesSystem.php
 *           │               │
 *           │               ├─<─ ещё константа вида MIMIMI_...
 *           │               ├─<─ процедура mimimiInclude()
 *           │               └─<─ и ещё несколько процедур вида mimimi...()
 *           │
 *           │   ┌─────────────────────────────────────┐
 *           ├─> │ СОЗДАНИЕ ГЛОБАЛЬНОЙ ПЕРЕМЕННОЙ $app │
 *           │   └─────────────────────────────────────┘
 *           │
 *           │                    ┌─> mimimi.core/RoutinesWeb.php
 *           │                    │               │
 *           │                    │               ├─<─ процедура mimimiUri()
 *           │                    │               ├─<─ процедура mimimiModule()
 *           │                    │               └─<─ и ещё несколько процедур вида mimimi...()
 *           │                    │
 *           │                    │                      ┌─<─ класс mimimi.core/Module.php
 *           │                    │                      │                             │
 *           │                    ├─<─ класс mimimi.core/NodeModule.php                ├─<─ свойство $owner
 *           │                    │                      │                             ├─<─ свойство $app
 *           └─> repost.vacancies/Application.php        ├─<─ свойство $myNodeFile     └─<─ метод    run()
 *                                     │                 └─<─ метод    getNodePath()
 *                                     └─> run()
 *                                          │
 *                              ┌───────────┴──────────┐
 *                              │ ЗАПУСК МОДУЛЯ СЕАНСА │
 *                              └───────────┬──────────┘                 ┌─<─ класс mimimi.core/Module.php
 *                                          │                            │
 *                                          ├─> repost.vacancies/Session/Session.php
 *                                          │                            │  │
 *                                          │                            │  └─> run()
 *                                          │                            │
 *                                          └─> routeFor()               ├─<─ метод get()
 *                                                  │                    ├─<─ метод set()
 *                                   ┌──────────────┴────────────┐       └─<─ метод remove()                ┌─<─ класс mimimi.core/Module.php
 *                                   │ ЗАПУСК МОДУЛЯ КОНТРОЛЛЕРА │                                          │
 *                                   └──────────────┬────────────┘                   ┌─<─ класс mimimi.core/NodeModule.php
 *                                                  │                                │                                                            ┌─<─ класс mimimi.core/Module.php
 *                                                  └─> repost.vacancies/Controllers/Controllers.php                                              │
 *                                                                                   │    │                                ┌───────────────────┐ ┌┴──────────────────┐
 *                                                                                   │    └─> repost.vacancies/Controllers/│ Некий контроллер: │/│ Некий контроллер: │.php
 *                                                                                   │                                     │     Dashboard     │ │     Dashboard     │
 *                                                                                   ├─<─ метод getPageNum()               │         Faq       │ │         Faq       │
 *                                                                                   ├─<─ метод getItemId()                │        Home       │ │        Home       │
 *                                                                                   └─<─ метод getFilter()                │       Parser      │ │       Parser      │
 *                                                                                                                         │         Seo       │ │         Seo       │
 *                                                                                                                         │      Vacancies    │ │      Vacancies    │
 *                                                                                                                         └───────────────────┘ └────────┬──────────┘
 *                                                                                                                                                        │
 *                                                                                                                                                        └─> run()
 *                                                                                                                                                             │
 *           ┌─────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────┘
 *           │
 *           │                                                                                                         ┌─<─ класс mimimi.core/Module.php
 *           │                                                                                                         │
 *           │   ┌────────────────────────────────────┐                                         ┌─<─ класс mimimi.core/NodeModule.php
 *           ├───┤ ИЗВЛЕЧЕНИЕ ДАННЫХ С ПОМОЩЬЮ МОДЕЛИ │                                         │
 *           │   └─────────────────┬──────────────────┘        ┌─<─ класс mimimi.modules/Sqlite/Sqlite.php
 *           │                     │                           │                                │
 *           │                     │                           │                                ├─<─ метод escape()
 *           │                     │                           │                                ├─<─ метод bind()
 *           │                     │                           │                                ├─<─ метод makeSET()
 *           │                     │                           │                                ├─<─ метод exec()
 *           │                     │                           │                                ├─<─ метод query()
 *           │                     │                           │                                ├─<─ метод queryFirst()
 *           │                     │                           │                                ├─<─ метод queryRow()
 *           │                     │                           │                                └─<─ метод queryRows()
 *           │                     │                           │
 *           │                     │                           │                                                 ┌─<─ класс mimimi.core/Module.php
 *           │                     └─> repost.vacancies/Models/Models.php                                        │
 *           │                                                 │  │                           ┌───────────────┐ ┌┴──────────────┐
 *           │                                                 │  └─> repost.vacancies/Models/│ Некая модель: │/│ Некая модель: │.php
 *           │                                                 │                              │    Settings   │ │    Settings   │
 *           │                                                 ├─<─ метод addFor()            │      Tasks    │ │      Tasks    │
 *           │                                                 ├─<─ метод getByFor()          │    Vacancies  │ │    Vacancies  │
 *           │                                                 ├─<─ метод selectAllFor()      └───────────────┘ └───────┬───────┘
 *           │                                                 ├─<─ метод removeFor()                                   │
 *           │                                                 └─<─ метод updateFor()                                   └─> некийМетодМодели()
 *           │   ┌──────────────────────────────────────────┐                            ┌─<─ класс mimimi.core/Module.php
 *           ├───┤ ГЕНЕРАЦИЯ ОТВЕТА С ПОМОЩЬЮ ПРЕДСТАВЛЕНИЯ │                            │
 *           │   └─────────────────────┬────────────────────┘     ┌─<─ класс mimimi.core/NodeModule.php
 *           │                         │                          │                                                      ┌─<─ класс mimimi.core/Module.php
 *           │                         └─> repost.vacancies/Views/Views.php                                              │
 *           │                                                      │                          ┌──────────────────────┐ ┌┴─────────────────────┐
 *           │                                                      └─> repost.vacancies/Views/│ Некое представление: │/│ Некое представление: │.php
 *           │                                                                                 │          Html        │ │          Html        │
 *           │                                                                                 │           Max        │ │           Max        │
 *           │                                                                                 │        Telegram      │ │        Telegram      │
 *           │   ┌────────────────────────────────────────────────┐                            └──────────────────────┘ └───────────┬──────────┘
 *           └─> │ ОТПРАВКА КОНЕЧНОГО HTML В БРАУЗЕР ПОЛЬЗОВАТЕЛЯ │                                                                 │
 *               └────────────────────────────────────────────────┘                                                                 └─> некийМетодПредставления()
 *
 * -------------------------------------------------------------------------
 *
 * @package    MimimiFramework
 * @subpackage Examples / Repost Vacancies
 * @license    GPL-2.0
 *             https://opensource.org/license/gpl-2-0/
 * @copyright  2022 MiMiMi Community
 *             https://mimimi.software/
 *
 * -------------------------------------------------------------------------
 */

    /**
     * ---------------------------------------------------------------------
     *
     * Подключаем из папки ядра фреймворка файл "mimimi.core/RoutinesWeb.php".
     * Там объявлены несколько мелких процедур, разработанных для веб
     * приложений.
     *
     * ---------------------------------------------------------------------
     */

    mimimiLoad ( 'RoutinesWeb.php' );

    /**
     * ---------------------------------------------------------------------
     *
     * Подключаем из папки ядра фреймворка файл "mimimi.core/NodeModule.php".
     * Там объявлен класс "MimimiNodeModule", который является простейшей
     * заготовкой для модулей, поддерживающих дочерние, вызываемые через
     * оператор Стрелка. Этот класс подходит как основа для реализуемого
     * ниже модуля.
     *
     * ---------------------------------------------------------------------
     */

    mimimiInclude ( 'NodeModule.php' );

    /**
     * ---------------------------------------------------------------------
     *
     * Создаём на основе класса той заготовки новый класс, в котором напишем
     * программный код текущего модуля. Обратите внимание как задано имя
     * нового класса - оно сложено из системного префикса "MyMimimi", и имени
     * текущего PHP-файла без расширения.
     *
     * ---------------------------------------------------------------------
     */

    class MyMimimiApplication extends MimimiNodeModule {

        /**
         * -----------------------------------------------------------------
         *
         * Метод: Сгенерировать страницу, запрошенную браузером.
         *
         * -----------------------------------------------------------------
         *
         * Работа приложения по генерации любой веб-страницы или документа
         * начинается с этого метода. Он является входной точкой приложения
         * и выполняет следующие действия: запускает модуль сеанса (чтобы
         * приложение чуть позже могло отслеживать авторизованного админа),
         * получает относительный URL запрошенной страницы, и выполняет
         * маршрутизацию для этого URL.
         *
         * Я не захотел делать модуль маршрутизатора, а просто сымитировал
         * его с помощью метода routeFor(), находящегося ниже.
         *
         * -----------------------------------------------------------------
         *
         * @public
         * @param   string  $params  (необязательный) В главном модуле этот параметр всегда равен пустой строке.
         * @return  void
         *
         * -----------------------------------------------------------------
         */

        public function run ( $params = '' ) {
            $this->session->run ( );
            $url = mimimiUri ( FALSE );
            $this->routeFor ( $url );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Метод: Маршрутизировать запрос к его исполнителю.
         *
         * -----------------------------------------------------------------
         *
         * Этот метод просто запускает конкретный контроллер в зависимости
         * от вида URL. Таким образом, метод обслуживает следующие URL-ы:
         *
         *     https://ваш.сайт/
         *
         *     https://ваш.сайт/faq
         *     https://ваш.сайт/faq/for-developers
         *
         *     https://ваш.сайт/parse
         *
         *     https://ваш.сайт/login
         *     https://ваш.сайт/logout
         *     https://ваш.сайт/password/change
         *
         *     https://ваш.сайт/settings
         *
         *     https://ваш.сайт/tasks
         *     https://ваш.сайт/tasks/add
         *     https://ваш.сайт/tasks/copy
         *     https://ваш.сайт/tasks/delete
         *     https://ваш.сайт/tasks/edit
         *
         *     https://ваш.сайт/vacancies/delete
         *     https://ваш.сайт/vacancies/edit
         *     https://ваш.сайт/vacancies
         *
         *     https://ваш.сайт/robots.txt
         *     https://ваш.сайт/sitemap.xml
         *
         * Остальные URL-ы считаются Ошибкой 404, поэтому маршрутизируются
         * на контроллер Seo.
         *
         * На первом этапе разработки приложения я неудачно придумал формат
         * URL страниц, его первый сегмент не равнялся именам контроллеров,
         * что привело к необходимости вручную писать такие маршрутные связи,
         * ещё и повторять их в "маршрутизаторе" самих контроллеров, так как
         * методы run() там играют роль внутренних маршрутизаторов. Ведь,
         * например, контроллеры Dashboard и Seo имеют внутри себя несколько
         * разветвляющихся маршрутов. А при попытке дополнить или изменить
         * маршруты теперь надо помнить, что такие же изменения требуется
         * прописать ещё и в маршрутизаторе приложения.
         *
         * -----------------------------------------------------------------
         *
         * @protected
         * @param   string  $url  Относительный URL запрошенной страницы.
         * @return  void
         *
         * -----------------------------------------------------------------
         */

        protected function routeFor ( $url ) {
            switch ( $url ) {
                case '':
                     $this->controllers->home->run ( );
                     break;
                case 'faq':
                case 'faq/for-developers':
                     $this->controllers->faq->run ( $url );
                     break;
                case 'parse':
                     $this->controllers->parser->run ( );
                     break;
                case 'login':
                case 'logout':
                case 'password/change':
                     $this->controllers->dashboard->login->run ( $url );
                     break;
                case 'settings':
                     $this->controllers->dashboard->settings->run ( );
                     break;
                case 'tasks':
                case 'tasks/add':
                case 'tasks/copy':
                case 'tasks/delete':
                case 'tasks/edit':
                     $this->controllers->dashboard->tasks->run ( $url );
                     break;
                case 'vacancies/delete':
                case 'vacancies/edit':
                     $this->controllers->dashboard->vacancies->run ( $url );
                     break;
                case 'vacancies':
                     $this->controllers->vacancies->run ( );
                     break;
                case 'robots.txt':
                case 'sitemap.xml':
                     $this->controllers->seo->run ( $url );
                     break;
                default:
                     $this->controllers->seo->run ( '404' );
            }
        }
    }
