<?php
/**
 * -------------------------------------------------------------------------
 *
 * Модуль для хранителя контроллеров.
 *
 * -------------------------------------------------------------------------
 *
 * Несмотря на то, что данный модуль служит лишь для коллекционирования
 * контроллеров в одной папке, здесь тем не менее размещена часть методов,
 * что универсальны для нескольких контроллеров сразу. Такими являются, как
 * правило, методы извлечения стандартной информации из URL-а запрошенной
 * страницы. Ведь подразумевается, что формат URL единый для всех страниц
 * сайта, а значит повторять это извлечение в каждом контроллере не имеет
 * смысла, проще вынести подобные методы на вышестоящий уровень.
 *
 * -------------------------------------------------------------------------
 *
 * @package    MimimiFramework
 * @subpackage Examples / Repost Vacancies
 * @license    GPL-2.0
 *             https://opensource.org/license/gpl-2-0/
 * @copyright  2022 MiMiMi Community
 *             https://mimimi.software/
 *
 * -------------------------------------------------------------------------
 */

    /**
     * ---------------------------------------------------------------------
     *
     * Подключаем из папки ядра фреймворка файл "mimimi.core/NodeModule.php".
     * Там объявлен класс "MimimiNodeModule", который является простейшей
     * заготовкой для модулей, поддерживающих дочерние, вызываемые через
     * оператор Стрелка. Этот класс подходит как основа для реализуемого
     * ниже модуля.
     *
     * ---------------------------------------------------------------------
     */

    mimimiInclude ( 'NodeModule.php' );

    /**
     * ---------------------------------------------------------------------
     *
     * Создаём на основе класса той заготовки новый класс, в котором напишем
     * программный код текущего модуля. Обратите внимание как задано имя
     * нового класса - оно сложено из системного префикса "MyMimimi", и имени
     * текущего PHP-файла без расширения.
     *
     * ---------------------------------------------------------------------
     */

    class MyMimimiControllers extends MimimiNodeModule {

        /**
         * -----------------------------------------------------------------
         *
         * Метод: Получить номер текущей страницы списка из URL запроса.
         *
         * -----------------------------------------------------------------
         *
         * Здесь мы подразумеваем, что пагинация всегда задана в URL-е вот
         * таким образом:
         *
         *     https://ваш.сайт/некий-урл?page=ЧИСЛО
         *
         * Поскольку текущий модуль фактически является owner-ом любого
         * дочернего модуля, Вы можете оттуда обратиться к этому методу вот
         * так:
         *
         *     $num = $this->owner->getPageNum();
         *     или
         *     $num = $this->app->controllers->getPageNum();
         *
         * -----------------------------------------------------------------
         *
         * @public
         * @return  int  Извлечённый номер страницы. Исчисляется от 1 и выше.
         *
         * -----------------------------------------------------------------
         */

        public function getPageNum ( ) {
            $num = intval ( mimimiGet( 'page' ) );
            return max ( 1, $num );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Метод: Получить ИД затребованной записи из URL запроса.
         *
         * -----------------------------------------------------------------
         *
         * Здесь мы подразумеваем, что идентификатор записи всегда задан в
         * URL-е вот таким образом:
         *
         *     https://ваш.сайт/некий-урл?item=ИДЕНТИФИКАТОР
         *
         * Поскольку текущий модуль фактически является owner-ом любого
         * дочернего модуля, Вы можете оттуда обратиться к этому методу вот
         * так:
         *
         *     $id = $this->owner->getItemId();
         *     или
         *     $id = $this->app->controllers->getItemId();
         *
         * -----------------------------------------------------------------
         *
         * @public
         * @param   bool        $asInteger  (необязательный) TRUE  если ИД ожидается числом.
         *                                                   FALSE если ИД может быть строковым.
         * @return  string|int              Извлечённый идентификатор.
         *
         * -----------------------------------------------------------------
         */

        public function getItemId ( $asInteger = TRUE ) {
            $id = mimimiGet( 'item' );
            return $asInteger ? ( is_numeric ( $id ) ? intval ( $id )
                                                     : 0 )
                              : $id;
        }

        /**
         * -----------------------------------------------------------------
         *
         * Метод: Получить затребованный фильтр из URL запроса.
         *
         * -----------------------------------------------------------------
         *
         * Здесь мы подразумеваем, что фильтр записей всегда задан в URL-е
         * вот таким образом:
         *
         *     https://ваш.сайт/некий-урл?filter=ЗНАЧЕНИЕ
         *
         * Поскольку текущий модуль фактически является owner-ом любого
         * дочернего модуля, Вы можете оттуда обратиться к этому методу вот
         * так:
         *
         *     $filter = $this->owner->getFilter();
         *     или
         *     $filter = $this->app->controllers->getFilter();
         *
         * -----------------------------------------------------------------
         *
         * @public
         * @return  string  Извлечённое значение фильтра.
         *
         * -----------------------------------------------------------------
         */

        public function getFilter ( ) {
            return mimimiGet ( 'filter' );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Инициализируем симулятор "неймспейса", так как в силу конструкции
         * класса "MimimiNodeModule" такое действие предписано совершать в
         * каждом наследуемом классе, если начиная с его узла тоже требуется
         * поддержать способность обращаться к его дочерним модулям через
         * оператор Стрелка (->).
         *
         * Как Вы можете видеть, в папке этого модуля находятся несколько
         * контроллеров: Dashboard, Home, Parser, Seo, Vacancies, к которым
         * нам придётся обращаться в любом случае. Поэтому мы и выполнили
         * сейчас инициализацию "неймспейса".
         *
         * Под неймспейсом понимается уникализация имён вложенных модульных
         * классов. Посмотрите как назван относительно текущего класса любой
         * класс дочернего модуля.
         *
         * -----------------------------------------------------------------
         *
         * @var    string
         * @access protected
         *
         * -----------------------------------------------------------------
         */

        protected $myNodeFile = __FILE__;
    }
