<?php
/**
 * -------------------------------------------------------------------------
 *
 * Модуль для хранителя моделей.
 *
 * -------------------------------------------------------------------------
 *
 * Несмотря на то, что данный модуль служит лишь для коллекционирования
 * моделей в одной папке, здесь тем не менее размещена часть методов, что
 * универсальны для нескольких моделей сразу. Такими являются, как правило,
 * методы извлечения записей из произвольной таблицы базы данных. Ведь
 * подразумевается, что механизм извлечения единый для всех таблиц базы, а
 * значит воспроизводить этот механизм в каждой модели не имеет смысла,
 * проще вынести подобные методы на вышестоящий уровень.
 *
 * -------------------------------------------------------------------------
 *
 * @package    MimimiFramework
 * @subpackage Examples / Repost Vacancies
 * @license    GPL-2.0
 *             https://opensource.org/license/gpl-2-0/
 * @copyright  2022 MiMiMi Community
 *             https://mimimi.software/
 *
 * -------------------------------------------------------------------------
 */

    /**
     * ---------------------------------------------------------------------
     *
     * Подключаем из папки универсальных базовых модулей фреймворка файл
     * "mimimi.modules/Sqlite/Sqlite.php". Там объявлен класс "MimimiSqlite",
     * который является простым модулем для работы с SQL базой с помощью
     * библиотеки SQLite. Этот класс подходит как основа для реализуемого
     * ниже модуля.
     *
     * ---------------------------------------------------------------------
     */

    mimimiInclude ( 'Sqlite/Sqlite.php' );

    /**
     * ---------------------------------------------------------------------
     *
     * Создаём на основе того базового класса новый класс, в котором напишем
     * программный код текущего модуля. Обратите внимание как задано имя
     * нового класса - оно сложено из системного префикса "MyMimimi", и имени
     * текущего PHP-файла без расширения.
     *
     * ---------------------------------------------------------------------
     */

    class MyMimimiModels extends MimimiSqlite {

        /**
         * -----------------------------------------------------------------
         *
         * Метод: Добавить запись в таблицу.
         *
         * -----------------------------------------------------------------
         *
         * Поскольку текущий модуль фактически является owner-ом любого
         * дочернего модуля, Вы можете оттуда обратиться к этому методу вот
         * так:
         *
         *     $this->owner->addFor('ИМЯ_ТАБЛИЦЫ', $row);
         *     или
         *     $this->app->models->addFor('ИМЯ_ТАБЛИЦЫ', $row);
         *
         * -----------------------------------------------------------------
         *
         * @public
         * @param   string  $table  Имя таблицы, в которую произойдёт добавление.
         * @param   array   $row    Массив, индексированный именами колонок сохраняемой записи.
         * @return  void
         *
         * -----------------------------------------------------------------
         */

        public function addFor ( $table, $row ) {
            $this->exec ( 'INSERT INTO `' . $table . '` '
                        . '( ' .       $this->makeSET ( $row, TRUE,  FALSE, FALSE ) . ' ) '
                        . 'VALUES( ' . $this->makeSET ( $row, FALSE, FALSE        ) . ' )' );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Метод: Получить запись из таблицы по её ИД.
         *
         * -----------------------------------------------------------------
         *
         * Поскольку текущий модуль фактически является owner-ом любого
         * дочернего модуля, Вы можете оттуда обратиться к этому методу вот
         * так:
         *
         *     $row = $this->owner->getByFor('ИМЯ_ТАБЛИЦЫ', $id);
         *     или
         *     $row = $this->app->models->getByFor('ИМЯ_ТАБЛИЦЫ', $id);
         *
         * -----------------------------------------------------------------
         *
         * @public
         * @param   string  $table  Имя таблицы, из которой произойдёт чтение.
         * @param   int     $id     Идентификатор запрашиваемой записи.
         * @return  array           Массив, индексированный именами колонок извлечённой записи.
         *
         * -----------------------------------------------------------------
         */

        public function getByFor ( $table, $id ) {
            return $this->queryRow ( 'SELECT * '
                                   . 'FROM `' . $table . '` '
                                   . 'WHERE `id` = ' . $this->bind ( $id ) );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Метод: Получить всех список записей таблицы.
         *
         * -----------------------------------------------------------------
         *
         * Поскольку текущий модуль фактически является owner-ом любого
         * дочернего модуля, Вы можете оттуда обратиться к этому методу вот
         * так:
         *
         *     $row = $this->owner->selectAllFor('ИМЯ_ТАБЛИЦЫ');
         *     или
         *     $row = $this->app->models->selectAllFor('ИМЯ_ТАБЛИЦЫ');
         *
         * -----------------------------------------------------------------
         *
         * @public
         * @param   string  $table  Имя таблицы, из которой произойдёт чтение.
         * @return  array           Массив массивов, индексированных именами колонок извлечённых записей.
         *
         * -----------------------------------------------------------------
         */

        public function selectAllFor ( $table ) {
            return $this->queryRows ( 'SELECT * '
                                    . 'FROM `' . $table . '`' );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Метод: Удалить запись из таблицы по её ИД.
         *
         * -----------------------------------------------------------------
         *
         * Поскольку текущий модуль фактически является owner-ом любого
         * дочернего модуля, Вы можете оттуда обратиться к этому методу вот
         * так:
         *
         *     $this->owner->removeFor('ИМЯ_ТАБЛИЦЫ', $id);
         *     или
         *     $this->app->models->removeFor('ИМЯ_ТАБЛИЦЫ', $id);
         *
         * -----------------------------------------------------------------
         *
         * @public
         * @param   string  $table  Имя таблицы, в которой произойдёт удаление.
         * @param   int     $id     Идентификатор удаляемой записи.
         * @return  void
         *
         * -----------------------------------------------------------------
         */

        public function removeFor ( $table, $id ) {
            $this->exec ( 'DELETE '
                        . 'FROM `' . $table . '` '
                        . 'WHERE `id` = ' . $this->bind ( $id ) );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Метод: Обновить запись таблицу по её ИД.
         *
         * -----------------------------------------------------------------
         *
         * Поскольку текущий модуль фактически является owner-ом любого
         * дочернего модуля, Вы можете оттуда обратиться к этому методу вот
         * так:
         *
         *     $this->owner->updateFor('ИМЯ_ТАБЛИЦЫ', $id, $row);
         *     или
         *     $this->app->models->updateFor('ИМЯ_ТАБЛИЦЫ', $id, $row);
         *
         * -----------------------------------------------------------------
         *
         * @public
         * @param   string  $table  Имя таблицы, в которой произойдёт обновление.
         * @param   int     $id     Идентификатор обновляемой записи.
         * @param   array   $row    Массив, индексированный именами колонок сохраняемой записи.
         * @return  void
         *
         * -----------------------------------------------------------------
         */

        public function updateFor ( $table, $id, $row ) {
            $this->exec ( 'UPDATE `' . $table . '` '
                        . 'SET ' .          $this->makeSET ( $row ) . ' '
                        . 'WHERE `id` = ' . $this->bind    ( $id  ) );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Инициализируем симулятор "неймспейса", так как в силу конструкции
         * класса "MimimiSqlite" такое действие предписано совершать в
         * порождённом от него классе, чтобы сохранить способность обращаться
         * к его дочерним модулям через оператор Стрелка (->) и чтобы путь
         * хранения базы данных был привязан к папке текущему модулю, а не
         * его базового прототипа.
         *
         * Как Вы можете видеть, в папке этого модуля находятся несколько
         * моделей: Settings, Tasks, Vacancies, к которым нам придётся
         * обращаться в любом случае. Поэтому мы и выполнили сейчас
         * инициализацию "неймспейса".
         *
         * Под неймспейсом понимается уникализация имён вложенных модульных
         * классов. Посмотрите как назван относительно текущего класса любой
         * класс дочернего модуля.
         *
         * -----------------------------------------------------------------
         *
         * @var    string
         * @access protected
         *
         * -----------------------------------------------------------------
         */

        protected $myNodeFile = __FILE__;
    }
