<?php
/**
 * -------------------------------------------------------------------------
 *
 * The database module for selecting customer entries.
 *
 * -------------------------------------------------------------------------
 *
 * This module is always called using getById() method implemented below.
 * The initiator of that call is run() method of the main module of your
 * application.
 *
 * -------------------------------------------------------------------------
 *
 * @package    MimimiFramework
 * @subpackage Examples / Shopping UI with Vue.js
 * @license    GPL-2.0
 *             https://opensource.org/license/gpl-2-0/
 * @copyright  2022 MiMiMi Community
 *             https://mimimi.software/
 *
 * -------------------------------------------------------------------------
 */

    mimimiInclude ( 'ModuleWithTable.php' );
    class MyMimimiCustomers extends MimimiModuleWithTable {

        /**
         * -----------------------------------------------------------------
         *
         * Specify a name of the database table to store customers.
         *
         * -----------------------------------------------------------------
         *
         * @access public
         * @var    string
         *
         * -----------------------------------------------------------------
         */

        public $table = 'customers';

        /**
         * -----------------------------------------------------------------
         *
         * Define a database table structure.
         *
         * -----------------------------------------------------------------
         *
         * @access protected
         * @var    array
         *
         * -----------------------------------------------------------------
         */

        protected $tableFields = [
                      '`id`       BIGINT(20)    NOT NULL  AUTO_INCREMENT   COMMENT "customer system identifier"',
                      '`login`    VARCHAR(50)   NOT NULL                   COMMENT "login"',
                      '`hash`     VARCHAR(50)   NOT NULL                   COMMENT "password hash"',
                      '`name`     VARCHAR(50)   NOT NULL                   COMMENT "customer name"',
                      '`address`  VARCHAR(100)  NOT NULL                   COMMENT "delivery address"'
                  ];

        /**
         * -----------------------------------------------------------------
         *
         * Define a list of table keys to speed up the database operations
         * related to customers.
         *
         * -----------------------------------------------------------------
         *
         * @access protected
         * @var    array
         *
         * -----------------------------------------------------------------
         */

        protected $tableKeys = [
                      'PRIMARY KEY ( `id`    )',
                      'UNIQUE KEY  ( `login` )',
                      'KEY         ( `name`  )'
                  ];

        /**
         * -----------------------------------------------------------------
         *
         * Retrieves a customer entry by its identifier.
         *
         * -----------------------------------------------------------------
         *
         * Please note that "t1" below is an alias for the database table
         * "customers". Note also that the filter used below will actually
         * result in the following MySQL query:
         *
         *     SELECT t1.*
         *     FROM   customers AS t1
         *     WHERE  t1.id = $id
         *     LIMIT  1
         *
         * -----------------------------------------------------------------
         *
         * @public
         * @param   string      $id  The identifier of the customer you are looking for.
         * @return  array|bool       ARRAY on success. It contains an entry obtained from a database table.
         *                           FALSE on failure. This means no entries were found.
         *
         * -----------------------------------------------------------------
         */

        public function getById ( $id ) {
            $filter = [
                'select' => [ 't1.*'  => TRUE ],
                /* where */   't1.id' => $id
            ];
            return $this->select ( $filter );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Renames an entry column and/or changes its value.
         *
         * -----------------------------------------------------------------
         *
         * @protected
         * @param   string      $name   The name of entry column that may be renamed.
         * @param   mixed       $value  The value of the column that may be changed.
         * @return  void
         *
         * -----------------------------------------------------------------
         */

        protected function renameField ( & $name, & $value ) {
            switch ( $name ) {
                case 'password':
                     $name  = 'hash';
                     $value = $this->app->hashPassword ( $value );
                     break;
            }
        }

        /**
         * -----------------------------------------------------------------
         *
         * Specify demo rows that will be used as default customer entries
         * if the database does not have a table named "customers". In this
         * case, all demo rows will be automatically added to the newly
         * created primary table.
         *
         * -----------------------------------------------------------------
         *
         * @access protected
         * @var    array
         *
         * -----------------------------------------------------------------
         */

        protected $demoRows = [
            [
                'id'       => 1,
                'login'    => 'demo',
                'password' => '123',            // <-- it will be converted to a hash (see method renameField() above for details)
                'name'     => 'Demo Customer',
                'address'  => 'Demo Town street, house 123'
            ]
        ];
    };
