<?php
/**
 * -------------------------------------------------------------------------
 *
 * Snippet for rendering a product edit form.
 *
 * -------------------------------------------------------------------------
 *
 * These global variables are available:
 *
 *     $params = database entry retrieved by the "Vendors" module, it is an array consisting of these elements:
 *                   'id'           -->  system identifier
 *                   'vendor_id'    -->  vendor identifier
 *                   'image'        -->  image URL relative to the site URL,                 for example media/hello-world/product-1.jpg
 *                   'name'         -->  product name
 *                   'description'  -->  product text description
 *                   'price'        -->  product price,                                      for example 100.65
 *                   'amount'       -->  stock amount
 *                   'color'        -->  background color if product image has transparency, for example #2196f3
 *     $app    = your application
 *
 * -------------------------------------------------------------------------
 */

    /**
     * ---------------------------------------------------------------------
     *
     * Execute server logic according to this form.
     *
     * ---------------------------------------------------------------------
     */

    $error   = '';
    $success = '';

    $entry   = $params;

    if ( getFormInput ( 'submit' ) ) {

        /**
         * -----------------------------------------------------------------
         *
         * Check for an attack based on the ID field.
         *
         * -----------------------------------------------------------------
         */

        $pid = $params[ 'id'        ];
        $vid = $params[ 'vendor_id' ];

        if ( getFormInput ( 'id'        ) == $pid
        &&   getFormInput ( 'vendor_id' ) == $vid ) {

            /**
             * -------------------------------------------------------------
             *
             * Get the entered form fields as an entry in the database.
             *
             * -------------------------------------------------------------
             */

            $entry = [
                'vendor_id'   => $vid,
                'image'       => makeSecureUrl (
                                     getFormInput     ( 'image'       ), FALSE
                                 ),
                'name'        => getFormInput         ( 'name'        ),
                'description' => getFormInput         ( 'description' ),
                'price'       => round ( getFormInput ( 'price'       ), 2 ),
                'amount'      => round ( getFormInput ( 'amount'      )    ),
                'color'       => getFormInput         ( 'color'       )
            ];

            /**
             * -------------------------------------------------------------
             *
             * The product name cannot be empty.
             *
             * -------------------------------------------------------------
             */

            if ( $entry[ 'name' ] != '' ) {

                /**
                 * ---------------------------------------------------------
                 *
                 * Validate the price and quantity.
                 *
                 * ---------------------------------------------------------
                 */

                if ( $entry[ 'price'  ] >= 0
                &&   $entry[ 'amount' ] >= 0 ) {

                    /**
                     * -----------------------------------------------------
                     *
                     * Save changes to the database.
                     *
                     * -----------------------------------------------------
                     */

                    if ( $app->onProduction ) {
                        $pid ? $app->vendors->products->update ( $pid, $entry )
                             : $app->vendors->products->add    (       $entry );

                        /**
                         * -------------------------------------------------
                         *
                         * Redirect to the vendor's page.
                         *
                         * -------------------------------------------------
                         */

                        $url = printSiteUrl ( FALSE ) . mimimiUri ( FALSE );
                        $url = preg_replace ( '~/(add|\d+/edit)$~ui', '', $url );
                        mimimiStop ( $url, 307 );

                    /**
                     * -----------------------------------------------------
                     *
                     * Otherwise, prepare an error message.
                     *
                     * -----------------------------------------------------
                     */

                    } else {
                        $success = 'Everything is fine! However, this website did not save your changes because you are currently in demo mode. In this mode, any addition/change/deletion of pages is simply simulated to show you how it would work on a real site.';
                    }
                } else {
                   $error = 'The Product Price or Stock Amount field cannot be negative.';
                }
            } else {
               $error = 'The Product Name field is required.';
            }
            $entry[ 'id' ] = $pid;
        } else {
            $error = 'Your request cannot be accepted because an attempt to spoof the ID field has been detected.';
        }
    }

    /**
     * ---------------------------------------------------------------------
     *
     * Generate form content.
     *
     * ---------------------------------------------------------------------
     */

?>  <p class="description error"><?php   printValue ( $error   ) ?></p>
    <p class="description success"><?php printValue ( $success ) ?></p>

    <legend>
        <small>Product Name:</small>
        <input type="text" name="name" value="<?php printValue ( $entry[ 'name' ] ) ?>" maxlength="60" required>
    </legend>

    <legend>
        <small>Image URL Relative To Site URL:</small>
        <input type="text" name="image" value="<?php printValue ( $entry[ 'image' ] ) ?>" maxlength="255" pattern="((https?:)?//[\w\-][\w.\-]*?[\w\-]/)?([\w,.+\-][\w\/,.\+\-]*?[\w\-])?">
        <small>For example <code>media/hello-world/image.jpg</code>. Or if you want to use an absolute locator, enter it as <code>https://example.com/image.jpg</code>, or as <code>//example.com/image.jpg</code>.</small>
    </legend>

    <legend>
        <small>Background Color:</small>
        <input type="text" name="color" value="<?php printValue ( $entry[ 'color' ] ) ?>" maxlength="20" pattern="[a-zA-Z]+|#[a-fA-F\d]{3}|#[a-fA-F\d]{6}|#[a-fA-F\d]{8}">
        <small>Use it only if your product image has a transparent background. Otherwise, leave it blank. <br><br>
               For example:                               <br>
               by color name <code>blanchedalmond</code>, <br>
               by HEX code <code>#ffebcd</code>,          <br>
               with alpha channel <code>#ffebcd44</code>, <br>
               in short notation <code>#fec</code>.</small>
    </legend>

    <legend>
        <small>Product Description:</small>
        <textarea name="description" rows="5" maxlength="1024"><?php printValue ( $entry[ 'description' ] ) ?></textarea>
    </legend>

    <legend>
        <small>Product Price:</small>
        <input type="text" name="price" value="<?php printValue ( $entry[ 'price' ] === ''
                                                                                      ? ''
                                                                                      : round ( $entry[ 'price' ], 2 ) ) ?>" required pattern="(0|[1-9]\d*)(\.\d\d?)?">
        <small>For example <code>25.64</code></small>
    </legend>

    <legend>
        <small>Stock Amount:</small>
        <input type="number" name="amount" value="<?php printValue ( $entry[ 'amount' ] ) ?>" min="0" step="1" max="50000">
    </legend>

    <input type="hidden" name="id"        value="<?php printValue ( $entry[ 'id'        ] ) ?>">
    <input type="hidden" name="vendor_id" value="<?php printValue ( $entry[ 'vendor_id' ] ) ?>">

    <div class="item">
        <div class="item-block">
            <nav class="bottom-area hr">
                <div class="price">&nbsp;</div>
                <button class="button" type="submit" name="submit" value="1">
                    <p>SAVE CHANGES</p>
                </button>
            </nav>
        </div>
    </div>
